<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Plugin\Actionlog\Ats\Extension;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\Permissions;
use Akeeba\Component\ATS\Administrator\Table\ManagernoteTable;
use Akeeba\Component\ATS\Administrator\Table\PostTable;
use Akeeba\Component\ATS\Administrator\Table\TicketTable;
use Doctrine\Inflector\Rules\English\InflectorFactory;
use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\Table\Table;
use Joomla\CMS\User\User;
use Joomla\Component\Actionlogs\Administrator\Plugin\ActionLogPlugin;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;
use ReflectionClass;
use ReflectionMethod;

class Ats extends ActionLogPlugin implements SubscriberInterface
{
	/**
	 * Load the language file on instantiation.
	 *
	 * @var    boolean
	 * @since  5.0.0
	 */
	protected $autoloadLanguage = true;

	private $defaultExtension = 'com_ats';

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return  array
	 *
	 * @since   5.0.0
	 */
	public static function getSubscribedEvents(): array
	{
		// Only subscribe events if the component is installed and enabled
		if (!ComponentHelper::isEnabled('com_ats'))
		{
			return [];
		}

		// Register all public onSomething methods as event handlers
		$events   = [];
		$refClass = new ReflectionClass(__CLASS__);
		$methods  = $refClass->getMethods(ReflectionMethod::IS_PUBLIC);

		foreach ($methods as $method)
		{
			$name = $method->getName();

			if (substr($name, 0, 2) != 'on')
			{
				continue;
			}

			$events[$name] = $name;
		}

		return $events;
	}

	/**
	 * An attachment has been deleted.
	 *
	 * Note that on deletion we cannot log any information about the attachment since it's, well, already deleted!
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsAttachmentsControllerAfterDelete(Event $event): void
	{
		$this->logCRUDDelete($event, 'PLG_ACTIONLOG_ATS_ATTACHMENT_DELETE');
	}

	/**
	 * An attachment has been published.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsAttachmentsControllerAfterPublish(Event $event): void
	{
		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_ATTACHMENT_PUBLISH', function ($table, &$langKey) {
			$post   = $table->getPost();
			$ticket = $post->getTicket();

			return [
				'postid'      => $post->getId(),
				'postlink'    => sprintf('index.php?option=com_ats&view=post&task=edit&id=%u', $post->getId()),
				'ticketid'    => $ticket->getId(),
				'ticketlink'  => sprintf('index.php?option=com_ats&view=ticket&task=edit&id=%u', $ticket->getId()),
				'tickettitle' => $ticket->title,
				'catid'       => $ticket->catid,
				'category'    => $ticket->getCategoryName(),
				'catlink'     => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		});
	}

	/**
	 * An attachment has been unpublished.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsAttachmentsControllerAfterUnpublish(Event $event): void
	{
		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_ATTACHMENT_UNPUBLISH', function ($table, &$langKey) {
			$post   = $table->getPost();
			$ticket = $post->getTicket();

			return [
				'postid'      => $post->getId(),
				'postlink'    => sprintf('index.php?option=com_ats&view=post&task=edit&id=%u', $post->getId()),
				'ticketid'    => $ticket->getId(),
				'ticketlink'  => sprintf('index.php?option=com_ats&view=ticket&task=edit&id=%u', $ticket->getId()),
				'tickettitle' => $ticket->title,
				'catid'       => $ticket->catid,
				'category'    => $ticket->getCategoryName(),
				'catlink'     => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		});
	}

	/**
	 * An attachment is about to be downloaded.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsAttachmentsControllerBeforeDownload(Event $event): void
	{
		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_ATTACHMENT_DOWNLOAD', function ($table, &$langKey) {
			$post   = $table->getPost();
			$ticket = $post->getTicket();

			return [
				'postid'      => $post->getId(),
				'postlink'    => sprintf('index.php?option=com_ats&view=post&task=edit&id=%u', $post->getId()),
				'ticketid'    => $ticket->getId(),
				'ticketlink'  => sprintf('index.php?option=com_ats&view=ticket&task=edit&id=%u', $ticket->getId()),
				'tickettitle' => $ticket->title,
				'catid'       => $ticket->catid,
				'category'    => $ticket->getCategoryName(),
				'catlink'     => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		});
	}

	/**
	 * The user finished linking an email service to Akeeba Ticket System
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	public function onComAtsAuthenticateControllerAfterCallback(Event $event): void
	{
		$service = $this->getApplication()->getInput()->getCmd('service', '');
		$key     = strtoupper('PLG_ACTIONLOG_ATS_AUTHENTICATE_CALLBACK_' . $service);
		$key     = Text::_($key) !== $key ? $key : 'PLG_ACTIONLOG_ATS_AUTHENTICATE_CALLBACK';

		$this->logUserAction([
			'service' => $service,
		], $key);
	}

	/**
	 * The user finished linking an email service to Akeeba Ticket System
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	public function onComAtsAuthenticateControllerAfterConset(Event $event): void
	{
		$service = $this->getApplication()->getInput()->getCmd('service', '');
		$key     = strtoupper('PLG_ACTIONLOG_ATS_AUTHENTICATE_CONSENT_' . $service);
		$key     = Text::_($key) !== $key ? $key : 'PLG_ACTIONLOG_ATS_AUTHENTICATE_CONSENT';

		$this->logUserAction([
			'service' => $service,
		], $key);
	}

	/**
	 * An Automatic Reply has been deleted.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsAutorepliesControllerAfterDelete(Event $event): void
	{
		$this->logCRUDDelete($event, 'PLG_ACTIONLOG_ATS_AUTOREPLY_DELETE');
	}

	/**
	 * An Automatic Reply has been published.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsAutorepliesControllerAfterPublish(Event $event): void
	{
		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_AUTOREPLY_PUBLISH', null, 'title');
	}

	/**
	 * An Automatic Reply has been unpublished.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsAutorepliesControllerAfterUnpublish(Event $event): void
	{
		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_AUTOREPLY_UNPUBLISH', null, 'title');
	}

	/**
	 * An Automatic Reply has been created or edited. The form is reloaded.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsAutoreplyControllerAfterApply(Event $event): void
	{
		$this->onComAtsAutoreplyControllerAfterSave($event);
	}

	/**
	 * An Automatic Reply has been created or edited. The form is closed.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsAutoreplyControllerAfterSave(Event $event): void
	{
		$this->logCRUDSave($event, [
			'create' => 'PLG_ACTIONLOG_ATS_AUTOREPLY_CREATE',
			'edit'   => 'PLG_ACTIONLOG_ATS_AUTOREPLY_EDIT',
		], null, 'title');
	}

	/**
	 * A Canned Reply has been deleted.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsCannedrepliesControllerAfterDelete(Event $event): void
	{
		$this->logCRUDDelete($event, 'PLG_ACTIONLOG_ATS_CANNEDREPLY_DELETE');
	}

	/**
	 * A Canned Reply has been published.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsCannedrepliesControllerAfterPublish(Event $event): void
	{
		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_CANNEDREPLY_PUBLISH', null, 'title');
	}

	/**
	 * A Canned Reply has been unpublished.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsCannedrepliesControllerAfterUnpublish(Event $event): void
	{
		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_CANNEDREPLY_UNPUBLISH', null, 'title');
	}

	/**
	 * A Canned Reply has been created or edited. The form is reloaded.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsCannedreplyControllerAfterApply(Event $event): void
	{
		$this->onComAtsCannedreplyControllerAfterSave($event);
	}

	/**
	 * A Canned Reply has been created or edited. The form is closed.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsCannedreplyControllerAfterSave(Event $event): void
	{
		$this->logCRUDSave($event, [
			'create' => 'PLG_ACTIONLOG_ATS_CANNEDREPLY_CREATE',
			'edit'   => 'PLG_ACTIONLOG_ATS_CANNEDREPLY_EDIT',
		], null, 'title');
	}

	/**
	 * A Manager's Note has been created or edited. The form is reloaded.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsManagernoteControllerAfterApply(Event $event): void
	{
		$this->onComAtsManagernoteControllerAfterSave($event);
	}

	/**
	 * A Manager's Note has been created or edited. The form is closed.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsManagernoteControllerAfterSave(Event $event): void
	{
		$this->logCRUDSave($event, [
			'create' => 'PLG_ACTIONLOG_ATS_MANAGERNOTE_CREATE_OWN',
			'edit'   => 'PLG_ACTIONLOG_ATS_MANAGERNOTE_EDIT_OWN',
		], function (ManagernoteTable $table, &$langKey) {
			$ticket = $table->getTicket();

			return [
				'ticketid'    => $ticket->getId(),
				'ticketlink'  => sprintf('index.php?option=com_ats&view=ticket&task=edit&id=%u', $ticket->getId()),
				'tickettitle' => $ticket->title,
				'catid'       => $ticket->catid,
				'category'    => $ticket->getCategoryName(),
				'catlink'     => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		});
	}

	/**
	 * A Manager's Note has been deleted.
	 *
	 * Note that on deletion we cannot log any information about the Manager's Note since it's, well, already deleted!
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsManagernotesControllerAfterDelete(Event $event): void
	{
		$this->logCRUDDelete($event, 'PLG_ACTIONLOG_ATS_MANAGERNOTE_DELETE');
	}

	/**
	 * A post has been created or edited. The form is reloaded.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsPostControllerAfterApply(Event $event): void
	{
		$this->onComAtsTicketControllerAfterSave($event);
	}

	/**
	 * A post has been created or edited. The form is closed.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsPostControllerAfterSave(Event $event): void
	{
		$this->logCRUDSave($event, [
			'create' => 'PLG_ACTIONLOG_ATS_POST_CREATE',
			'edit'   => 'PLG_ACTIONLOG_ATS_POST_EDIT',
		], function (PostTable $table, &$langKey) {
			$ticket = $table->getTicket();

			return [
				'ticketid'    => $ticket->getId(),
				'ticketlink'  => sprintf('index.php?option=com_ats&view=ticket&task=edit&id=%u', $ticket->getId()),
				'tickettitle' => $ticket->title,
				'catid'       => $ticket->catid,
				'category'    => $ticket->getCategoryName(),
				'catlink'     => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		});
	}

	/**
	 * A post has been deleted.
	 *
	 * Note that on deletion we cannot log any information about the post since it's, well, already deleted!
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsPostsControllerAfterDelete(Event $event): void
	{
		$this->logCRUDDelete($event, 'PLG_ACTIONLOG_ATS_POST_DELETE');
	}

	/**
	 * A post has been published.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsPostsControllerAfterPublish(Event $event): void
	{
		$this->logCRUDSave(
			$event,
			'PLG_ACTIONLOG_ATS_POST_PUBLISH',
			function (PostTable $table, &$langKey) {
				$ticket = $table->getTicket();

				return [
					'ticketid'    => $ticket->getId(),
					'ticketlink'  => sprintf('index.php?option=com_ats&view=ticket&task=edit&id=%u', $ticket->getId()),
					'tickettitle' => $ticket->title,
					'catid'       => $ticket->catid,
					'category'    => $ticket->getCategoryName(),
					'catlink'     => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
				];
			}, 'title');
	}

	/**
	 * A post has been unpublished.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsPostsControllerAfterUnpublish(Event $event): void
	{
		$this->logCRUDSave(
			$event,
			'PLG_ACTIONLOG_ATS_POST_UNPUBLISH',
			function (PostTable $table, &$langKey) {
				$ticket = $table->getTicket();

				return [
					'ticketid'    => $ticket->getId(),
					'ticketlink'  => sprintf('index.php?option=com_ats&view=ticket&task=edit&id=%u', $ticket->getId()),
					'tickettitle' => $ticket->title,
					'catid'       => $ticket->catid,
					'category'    => $ticket->getCategoryName(),
					'catlink'     => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
				];
			}, 'title');
	}

	/**
	 * A ticket status was assigned
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsTicketControllerAfterAjax_set_assigned(Event $event): void
	{
		$assigned = $this->getApplication()->getInput()->getCmd('assigned', null);

		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_TICKET_SET_ASSIGNED',
			function ($table, &$langKey) use ($assigned) {
				$assignee = Permissions::getUser($assigned);

				return [
					'catid'        => $table->catid,
					'category'     => $table->getCategoryName(),
					'catlink'      => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
					'assigned'     => $assigned,
					'assignee'     => $assignee->username,
					'assigneelink' => sprintf('index.php?option=com_users&task=user.edit&id=%u', $assignee->id),
				];
			}, 'title');
	}

	/**
	 * A ticket status was changed
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsTicketControllerAfterAjax_set_status(Event $event): void
	{
		$status = $this->getApplication()->getInput()->getCmd('status', '-1');

		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_TICKET_SET_STATUS',
			function ($table, &$langKey) use ($status) {
				$statuses    = Permissions::getStatuses();
				$statusTitle = $statuses[strtoupper(trim($status))] ?? null;

				return [
					'catid'       => $table->catid,
					'category'    => $table->getCategoryName(),
					'catlink'     => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
					'status'      => $status,
					'statustitle' => $statusTitle,
				];
			}, 'title');
	}

	/**
	 * A ticket has been created or edited. The form is reloaded.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsTicketControllerAfterApply(Event $event): void
	{
		$this->onComAtsTicketControllerAfterSave($event);
	}

	/**
	 * One or more tickets have been batch processed
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsTicketControllerAfterBatch(Event $event): void
	{
		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_TICKET_BATCH', function ($table, &$langKey) {
			return [
				'catid'    => $table->catid,
				'category' => $table->getCategoryName(),
				'catlink'  => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		}, 'title');
	}

	/**
	 * A ticket has been created or edited. The form is closed.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsTicketControllerAfterSave(Event $event): void
	{
		$this->logCRUDSave($event, [
			'create.own'   => 'PLG_ACTIONLOG_ATS_TICKET_CREATE_OWN',
			'create.other' => 'PLG_ACTIONLOG_ATS_TICKET_CREATE_BEHALF',
			'edit.own'     => 'PLG_ACTIONLOG_ATS_TICKET_EDIT_OWN',
			'edit.other'   => 'PLG_ACTIONLOG_ATS_TICKET_EDIT_BEHALF',
		], function (TicketTable $table, &$langKey) {
			return [
				'catid'    => $table->catid,
				'category' => $table->getCategoryName(),
				'catlink'  => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		}, 'title');
	}

	/**
	 * A ticket has been closed.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsTicketsControllerAfterClose(Event $event): void
	{
		$this->logCRUDAction($event, [
			'own'   => 'PLG_ACTIONLOG_ATS_TICKET_CLOSE_OWN',
			'other' => 'PLG_ACTIONLOG_ATS_TICKET_CLOSE_BEHALF',
		], function ($table, &$langKey) {
			return [
				'catid'    => $table->catid,
				'category' => $table->getCategoryName(),
				'catlink'  => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		}, 'title');
	}

	/**
	 * A ticket has been deleted.
	 *
	 * Note that on deletion we cannot log any information about the ticket since it's, well, already deleted!
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsTicketsControllerAfterDelete(Event $event): void
	{
		$this->logCRUDDelete($event, 'PLG_ACTIONLOG_ATS_TICKET_DELETE');
	}

	/**
	 * A ticket has been made private.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsTicketsControllerAfterMakeprivate(Event $event): void
	{
		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_TICKET_MAKE_PRIVATE', function ($table, &$langKey) {
			return [
				'catid'    => $table->catid,
				'category' => $table->getCategoryName(),
				'catlink'  => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		}, 'title');
	}

	/**
	 * A ticket has been made public.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsTicketsControllerAfterMakepublic(Event $event): void
	{
		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_TICKET_MAKE_PUBLIC', function ($table, &$langKey) {
			return [
				'catid'    => $table->catid,
				'category' => $table->getCategoryName(),
				'catlink'  => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		}, 'title');
	}

	/**
	 * A ticket has been published.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsTicketsControllerAfterPublish(Event $event): void
	{
		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_TICKET_PUBLISH', function ($table, &$langKey) {
			return [
				'catid'    => $table->catid,
				'category' => $table->getCategoryName(),
				'catlink'  => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		}, 'title');
	}

	/**
	 * A ticket has been reopened.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsTicketsControllerAfterReopen(Event $event): void
	{
		$this->logCRUDAction($event, [
			'own'   => 'PLG_ACTIONLOG_ATS_TICKET_REOPEN_OWN',
			'other' => 'PLG_ACTIONLOG_ATS_TICKET_REOPEN_BEHALF',
		], function ($table, &$langKey) {
			return [
				'catid'    => $table->catid,
				'category' => $table->getCategoryName(),
				'catlink'  => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		}, 'title');
	}

	/**
	 * A ticket has been unpublished.
	 *
	 * @param   Event  $event  The Joomla event we are handling
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onComAtsTicketsControllerAfterUnpublish(Event $event): void
	{
		$this->logCRUDAction($event, 'PLG_ACTIONLOG_ATS_TICKET_UNPUBLISH', function ($table, &$langKey) {
			return [
				'catid'    => $table->catid,
				'category' => $table->getCategoryName(),
				'catlink'  => sprintf('index.php?option=com_categories&task=category.edit&id=%u&extension=com_ats', $table->catid),
			];
		}, 'title');
	}

	/**
	 * Gets the list of IDs from the request data
	 *
	 * @return  array
	 * @since   5.0.0
	 */
	private function getIDsFromRequest(): array
	{
		// Get the ID or list of IDs from the request or the configuration
		$cid = $this->getApplication()->getInput()->get('cid', [], 'array');
		$id  = $this->getApplication()->getInput()->getInt('id', 0);

		$ids = [];

		if (is_array($cid) && !empty($cid))
		{
			$ids = $cid;
		}
		elseif (!empty($id))
		{
			$ids = [$id];
		}

		return $ids;
	}

	/**
	 * Log a CRUD action
	 *
	 * @param   Event            $event            The Joomla event being handled
	 * @param   string|string[]  $translationKeys  The translation string keys for this user action.
	 * @param   callable|null    $callback         Optional callback to add info and/or change the translation key
	 * @param   string|null      $displayKey       The key of the table containing the displayed title
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function logCRUDAction(Event $event, $translationKeys, ?callable $callback = null, ?string $displayKey = null): void
	{
		$arguments = array_values($event->getArguments());
		/** @var BaseController $controller */
		$controller = $arguments[0];
		$ids        = $this->getIDsFromRequest();
		/** @var BaseDatabaseModel $model */
		$model = $controller->getModel();

		$controllerName = $controller->getName();
		$tableName      = $controller->getName();

		try
		{
			/** @var Table $table */
			$table = $model->getTable($tableName, 'Administrator');
		}
		catch (Exception $e)
		{
			$tableName = (new InflectorFactory())->build()->singularize($tableName);
			/** @var Table $table */
			$table = $model->getTable($tableName, 'Administrator');
		}

		foreach ($ids as $id)
		{
			if (!$table->load($id))
			{
				continue;
			}

			// Basic information about the record being acted upon
			$logData = [
				'id'        => $table->getId(),
				'title'     => ($displayKey ? $table->{$displayKey} : null) ?? sprintf('#%u', $table->getId()),
				'titlelink' => sprintf('index.php?option=com_ats&view=%s&task=edit&id=%u', $controllerName, $id),
			];

			// Evaluate “on behalf of” (creating / editing a record owned by a user other than our own)
			$canHaveOnBehalf = $table->hasField('created_by') && $table->hasField('modified_by');
			$onBehalf        = false;

			if ($canHaveOnBehalf)
			{
				$onBehalfUser           = Permissions::getUser($table->created_by);
				$onBehalf               = true;
				$logData['behalf']      = $onBehalfUser->username;
				$logData['behalf_id']   = $onBehalfUser->id;
				$logData['behalf_link'] = sprintf('index.php?option=com_users&task=user.edit&id=%u', $onBehalfUser->id);
			}

			// Select the correct translation key
			$logLangKey = $translationKeys;

			if (!is_string($translationKeys) && is_array($translationKeys))
			{
				$key = $canHaveOnBehalf ? ($onBehalf ? 'own' : 'other') : '';

				if (isset($translationKeys[$key]))
				{
					$logLangKey = $translationKeys[$key];
				}
				else
				{
					$logLangKey = array_shift($translationKeys);
				}
			}

			// Additional post–processing from the callback
			if (!empty($callback) && is_callable($callback))
			{
				$logData = array_merge($logData, call_user_func_array($callback, [$table, &$logLangKey]));
			}

			$this->logUserAction($logData, $logLangKey);
		}
	}

	/**
	 * Log a CRUD deletion
	 *
	 * @param   Event   $event           The Joomla event being handled
	 * @param   string  $translationKey  The translation string key for this user action.
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function logCRUDDelete(Event $event, string $translationKey): void
	{
		$arguments = array_values($event->getArguments());
		/** @var BaseController $controller */
		$controller = $arguments[0];
		$ids        = $this->getIDsFromRequest();
		/** @var BaseDatabaseModel $model */
		$model     = $controller->getModel();
		$tableName = $controller->getName();

		try
		{
			/** @var Table $table */
			$table = $model->getTable($tableName, 'Administrator');
		}
		catch (Exception $e)
		{
			$tableName = (new InflectorFactory())->build()->singularize($tableName);
			/** @var Table $table */
			$table = $model->getTable($tableName, 'Administrator');
		}

		foreach ($ids as $id)
		{
			// Basic information about the record being acted upon
			$logData = [
				'id'    => $id,
				'title' => sprintf('#%u', $table->getId()),
			];

			$this->logUserAction($logData, $translationKey);
		}
	}


	/**
	 * Log a CRUD save action
	 *
	 * @param   Event            $event            The Joomla event being handled
	 * @param   string|string[]  $translationKeys  The translation string keys for this user action.
	 * @param   callable|null    $callback         Optional callback to add info and/or change the translation key
	 * @param   string|null      $displayKey       The key of the table containing the displayed title
	 *
	 * @return  void
	 * @throws Exception
	 * @since   5.0.0
	 */
	private function logCRUDSave(Event $event, $translationKeys, ?callable $callback = null, ?string $displayKey = null): void
	{
		// Get the Controller, Model and Table
		$arguments = array_values($event->getArguments());
		/** @var BaseController $controller */
		$controller = $arguments[0];
		/** @var BaseDatabaseModel $model */
		$model          = $controller->getModel();
		$controllerName = $controller->getName();
		$tableName      = $controllerName;

		try
		{
			/** @var Table $table */
			$table = $model->getTable($tableName, 'Administrator');
		}
		catch (Exception $e)
		{
			$tableName = (new InflectorFactory())->build()->singularize($tableName);
			/** @var Table $table */
			$table = $model->getTable($tableName, 'Administrator');
		}

		$primaryKey = $table->getKeyName();

		// Is this a new record?
		$input = $this->getApplication()->getInput();
		$view  = $input->getCmd('view', null);
		$task  = $input->getCmd('task', 'save');

		if (strpos($task, '.') !== false)
		{
			[$view, $task] = explode('.', $task, 2);
		}

		$data      = $input->post->get('jform', [], 'array');
		$id        = (strtolower($view) === strtolower($controllerName)) ? $input->getInt('id', null) : null;
		$id        = $id ?: ($data[$primaryKey] ?? null);
		$newRecord = is_null($id);

		// Get the effective ID of the record saved
		$id = $id ?? $model->getState(sprintf('%s.id', strtolower($model->getName())));

		// Make sure the record can be loaded
		if (!$table->load($id))
		{
			return;
		}

		// Basic information about the record being saved
		$logData = [
			'id'        => $table->getId(),
			'title'     => ($displayKey ? $table->{$displayKey} : null) ?? sprintf('#%u', $table->getId()),
			'titlelink' => sprintf('index.php?option=com_ats&view=%s&task=edit&id=%u', $controllerName, $id),
		];

		// Evaluate “on behalf of” (creating / editing a record owned by a user other than our own)
		$canHaveOnBehalf = $table->hasField('created_by') && $table->hasField('modified_by');
		$onBehalf        = false;

		if ($canHaveOnBehalf)
		{
			$onBehalfUser           = Permissions::getUser($table->created_by);
			$onBehalf               = true;
			$logData['behalf']      = $onBehalfUser->username;
			$logData['behalf_id']   = $onBehalfUser->id;
			$logData['behalf_link'] = sprintf('index.php?option=com_users&task=user.edit&id=%u', $onBehalfUser->id);
		}

		// Select the correct translation key
		$logLangKey = $translationKeys;

		if (!is_string($translationKeys) && is_array($translationKeys))
		{
			$altKey = ($newRecord ? 'create' : 'edit');
			$key    = $canHaveOnBehalf ? ($altKey . '.' . ($onBehalf ? 'own' : 'other')) : $altKey;

			if (isset($translationKeys[$key]))
			{
				$logLangKey = $translationKeys[$key];
			}
			elseif (isset($translationKeys[$altKey]))
			{
				$logLangKey = $translationKeys[$altKey];
			}
			else
			{
				$logLangKey = array_shift($translationKeys);
			}
		}

		// Additional post–processing from the callback
		if (!empty($callback) && is_callable($callback))
		{
			try
			{
				$logData = array_merge($logData, call_user_func_array($callback, [$table, &$logLangKey]));
			}
			catch (\Throwable $e)
			{
				// Ignore
			}
		}

		$this->logUserAction($logData, $logLangKey);
	}

	/**
	 * Log a user action.
	 *
	 * This is a simple wrapper around self::addLog
	 *
	 * @param   string|array  $dataOrTitle         Text to use as the title, or an array of data to record in the audit
	 *                                             log.
	 * @param   string        $messageLanguageKey  Language key describing the user action taken.
	 * @param   string|null   $context             The name of the extension being logged.
	 * @param   User|null     $user                User object taking this action.
	 *
	 * @return  void
	 *
	 * @see     self::addLog
	 * @since   5.0.0
	 */
	private function logUserAction($dataOrTitle, string $messageLanguageKey, ?string $context = null, ?User $user = null): void
	{
		// Get the user if not defined
		$user = $user ?? $this->getApplication()->getIdentity();

		// No log for guests
		if (empty($user) || ($user->guest))
		{
			return;
		}

		// Default extension if none defined
		$context = $context ?? $this->defaultExtension;

		if (!is_array($dataOrTitle))
		{
			$dataOrTitle = [
				'title' => $dataOrTitle,
			];
		}

		$this->addLog([$dataOrTitle], $messageLanguageKey, $context, $user->id);
	}

}