<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Plugin\Finder\Tickets\Extension;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Table\TicketTable;
use Exception;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Categories\Categories;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\Table\Table;
use Joomla\Component\Fields\Administrator\Helper\FieldsHelper;
use Joomla\Component\Finder\Administrator\Indexer\Adapter;
use Joomla\Component\Finder\Administrator\Indexer\Helper;
use Joomla\Component\Finder\Administrator\Indexer\Indexer;
use Joomla\Component\Finder\Administrator\Indexer\Result;
use Joomla\Database\DatabaseQuery;
use Joomla\Database\ParameterType;
use Joomla\Database\QueryInterface;
use Joomla\Utilities\ArrayHelper;

/**
 * ATS Smart Search plugin
 */
class Tickets extends Adapter
{
	use RunPluginsTrait;

	/**
	 * The component's MVC factory
	 *
	 * @var   MVCFactoryInterface|null;
	 * @since 5.0.0
	 */
	private static $mvcFactory;

	/**
	 * Load the language file on instantiation.
	 *
	 * @var    boolean
	 * @since  5.0.0
	 */
	protected $autoloadLanguage = true;

	/**
	 * The plugin identifier.
	 *
	 * @var    string
	 * @since  5.0.0
	 */
	protected $context = 'Tickets';

	/**
	 * The component whose content is being indexed.
	 *
	 * @var    string
	 * @since  5.0.0
	 */
	protected $extension = 'com_ats';

	/**
	 * The component view to use when rendering the results.
	 *
	 * @var    string
	 * @since  5.0.0
	 */
	protected $layout = 'ticket';

	/**
	 * The field the published state is stored in.
	 *
	 * @var    string
	 * @since  5.0.0
	 */
	protected $state_field = 'enabled';

	/**
	 * The table name.
	 *
	 * @var    string
	 * @since  5.0.0
	 */
	protected $table = '#__ats_posts';

	/**
	 * The type of content that the adapter indexes.
	 *
	 * @var    string
	 * @since  5.0.0
	 */
	protected $type_title = 'Ticket';

	public function __construct(&$subject, $config, MVCFactoryInterface $factory, CMSApplicationInterface $app)
	{
		self::$mvcFactory = $factory;

		parent::__construct($subject, $config);

		$this->setApplication($app);

		$app->getLanguage()->load('lib_joomla');
	}


	/**
	 * Remove the link information for items that have been deleted..
	 *
	 * @param   string  $context  The context of the action being performed.
	 * @param   Table   $table    A JTable object containing the record to be deleted.
	 *
	 * @return  boolean  True on success.
	 *
	 * @throws  Exception on database error.
	 * @since   5.0.0
	 */
	public function onFinderAfterDelete($context, $table)
	{
		switch ($context)
		{
			case 'com_ats.post':
				/** @var  \Akeeba\Component\ATS\Administrator\Table\PostTable $table */
				$id = $table->id;
				break;

			case 'com_finder.index':
				/** @noinspection PhpUndefinedFieldInspection */
				$id = $table->link_id;
				break;

			default:
				return true;
		}

		// Remove the items.
		return $this->remove($id);
	}

	/**
	 * Method to determine if the visibility of a ticket changed.
	 *
	 * @param   string   $context  The context of the content passed to the plugin.
	 * @param   Table    $row      A JTable object
	 * @param   boolean  $isNew    If the content has just been created
	 *
	 * @return  boolean  True on success.
	 *
	 * @throws  Exception on database error.
	 * @since   5.0.0
	 */
	public function onFinderAfterSave($context, $row, $isNew)
	{
		// We only want to handle tickets and posts here
		if (!in_array($context, ['com_ats.post', 'com_ats.ticket']))
		{
			return true;
		}

		/** @var \Akeeba\Component\ATS\Administrator\Table\TicketTable $ticket */
		$ticket = $row;

		if ($context === 'com_ats.post')
		{
			/** @var \Akeeba\Component\ATS\Administrator\Table\PostTable $row */
			try
			{
				$ticket = $this->getFactory()->createTable('Ticket', 'Administrator');
			}
			catch (Exception $e)
			{
				return true;
			}

			if (!$ticket->load($row->ticket_id))
			{
				return true;
			}
		}

		if (!$ticket->public && !$isNew)
		{
			// Remove tickets converted from public to private
			$this->removeAll($ticket->id);
		}
		elseif ($context == 'com_ats.post')
		{
			// Reindex a public ticket's post
			$this->reindex($row->id);
		}

		return true;
	}

	/**
	 * Update the item link information when the ticket category changes.
	 *
	 * This is fired when the ticket category is published or unpublished from the list view.
	 *
	 * @param   string   $extension  The extension whose category has been updated.
	 * @param   array    $pks        A list of primary key ids of the content that has changed state.
	 * @param   integer  $value      The value of the state that the content has been changed to.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	public function onFinderCategoryChangeState($extension, $pks, $value)
	{
		// Make sure we're handling com_contact categories
		if ($extension === $this->extension)
		{
			$this->categoryStateChange($pks, $value);
		}
	}

	/**
	 * Method to update the link information for items that have been changed from outside the edit screen.
	 *
	 * This is fired when the item is published, unpublished, archived, or unarchived from the list view.
	 *
	 * @param   string   $context  The context for the content passed to the plugin.
	 * @param   array    $pks      A list of primary key ids of the content that has changed state.
	 * @param   integer  $value    The value of the state that the content has been changed to.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	public function onFinderChangeState($context, $pks, $value)
	{
		switch ($context)
		{
			// Post published / unpublished
			case 'com_ats.post':
				$this->itemStateChange($pks, $value);
				break;

			// Ticket made public / private
			case 'com_ats.ticket.public':
				$db    = method_exists($this, 'getDatabase') ? $this->getDatabase() : $this->db;
				$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
					->select($db->quoteName('id'))
					->from($db->quoteName('#__ats_posts'))
					->whereIn($db->quoteName('ticket_id'), $pks, ParameterType::INTEGER);

				$ids = $db->setQuery($query)->loadColumn();

				foreach ($ids as $post_id)
				{
					if ($value == 1)
					{
						// Ticket made public; reindex its posts.
						$this->reindex($post_id);
					}
					else
					{
						// Ticket made private; remove its posts from the index.
						$this->remove($post_id);
					}

				}
				break;

			// This plugin is enabled / disabled
			case 'com_plugins.plugin':
				if ($value === 0)
				{
					$this->pluginDisable($pks);
				}
				break;
		}
	}

	/**
	 * Method to get a content item to index.
	 *
	 * @param   integer  $id  The id of the content item.
	 *
	 * @return  Result  A Finder indexer Result object.
	 *
	 * @throws  Exception on database error.
	 * @since   5.0.0
	 */
	protected function getItem($id)
	{
		$db = method_exists($this, 'getDatabase') ? $this->getDatabase() : $this->db;

		// Get the list query and add the extra WHERE clause.
		$query = $this->getListQuery();
		$query->where($db->quoteName('a.id') . ' = :post_id')
			->bind(':post_id', $id, ParameterType::INTEGER);

		// Get the item to index.
		$db->setQuery($query);
		$row = $db->loadAssoc();

		// Convert the item to a result object.
		/** @var Result $item */
		$item = ArrayHelper::toObject($row, Result::class);

		// Set the item type.
		$item->type_id = $this->type_id;

		// Set the item layout.
		$item->layout = $this->layout;

		return $item;
	}

	/**
	 * Method to get the SQL query used to retrieve the list of content items.
	 *
	 * @param   DatabaseQuery|null  $query  A JDatabaseQuery object or null.
	 *
	 * @return  DatabaseQuery  A database object.
	 *
	 * @since   5.0.0
	 */
	protected function getListQuery($query = null)
	{
		$db = method_exists($this, 'getDatabase') ? $this->getDatabase() : $this->db;
		// Check if we can use the supplied SQL query.
		$query = $query instanceof QueryInterface
			? $query
			: (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true));

		$query->select(
			[
				$db->quoteName('a.id', 'id'),
				$db->quoteName('t.title'),
				$db->quoteName('t.alias'),
				$db->q('') . ' AS ' . $db->quoteName('summary'),
				$db->quoteName('a.content_html', 'body'),
				$db->quoteName('a.enabled'),
				$db->quoteName('t.catid'),
				$db->quoteName('a.created', 'start_date'),
				$db->quoteName('a.created_by'),
				$db->quoteName('a.modified'),
				$db->quoteName('a.modified_by'),
				$db->quoteName('a.ticket_id'),
				$db->quoteName('t.alias', 'slug'),
				$db->quoteName('t.public'),
				$db->quoteName('c.title', 'category'),
				$db->quoteName('c.published', 'cat_state'),
				$db->quoteName('c.access', 'cat_access'),
				$db->quoteName('c.language', 'language'),
				$db->quoteName('c.access', 'access'),
				$db->quoteName('u.name', 'author'),
			]
		);


		$case_when_category_alias = ' CASE WHEN ';
		$case_when_category_alias .= $query->charLength('c.alias');
		$case_when_category_alias .= ' THEN ';
		$c_id                     = $query->castAs('CHAR', 'c.id');
		$case_when_category_alias .= $query->concatenate([$c_id, 'c.alias'], ':');
		$case_when_category_alias .= ' ELSE ';
		$case_when_category_alias .= $c_id . ' END as catslug';
		$query->select($case_when_category_alias);

		$query
			->from($db->quoteName('#__ats_posts', 'a'))
			->leftJoin(
				$db->quoteName('#__ats_tickets', 't'),
				$db->quoteName('t.id') . ' = ' . $db->quoteName('a.ticket_id')
			)
			->leftJoin(
				$db->quoteName('#__categories', 'c'),
				$db->quoteName('c.id') . ' = ' . $db->quoteName('t.catid')
			)
			->leftJoin(
				$db->quoteName('#__users', 'u'),
				$db->quoteName('u.id') . ' = ' . $db->quoteName('a.created_by')
			)
			->where($db->quoteName('c.id') . ' IS NOT NULL');

		return $query;
	}

	/**
	 * Method to get a SQL query to load the published and access states for
	 * an article and category.
	 *
	 * @return  DatabaseQuery  A database object.
	 *
	 * @since   5.0.0
	 */
	protected function getStateQuery()
	{
		$db    = method_exists($this, 'getDatabase') ? $this->getDatabase() : $this->db;
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select(
				[
					$db->quoteName('a.id', 'id'),
					$db->quoteName('a.enabled', 'state'),
					$db->quoteName('c.published', 'cat_state'),
					$db->quoteName('c.access', 'cat_access'),
					$db->quoteName('c.access', 'access'),
				]
			)
			->from($db->quoteName($this->table, 'a'))
			->leftJoin(
				$db->quoteName('#__ats_tickets', 't'),
				$db->quoteName('t.id') . ' = ' . $db->quoteName('a.ticket_id')
			)
			->leftJoin(
				$db->quoteName('#__categories', 'c'),
				$db->quoteName('c.id') . ' = ' . $db->quoteName('t.catid')
			);

		return $query;
	}

	/**
	 * Method to get the URL for the item. The URL is how we look up the link in the Finder index.
	 *
	 * @param   integer  $id         The id of the item.
	 * @param   string   $extension  The extension the category is in.
	 * @param   string   $view       The view for the URL.
	 *
	 * @return  string  The URL of the item.
	 * @since   5.0.0
	 */
	protected function getUrl($id, $extension, $view)
	{
		if ($extension != $this->extension)
		{
			return parent::getUrl($id, $extension, $view);
		}

		/** @var \Akeeba\Component\ATS\Administrator\Table\PostTable $post */
		try
		{
			$post = $this->getFactory()->createTable('Post', 'Administrator');
		}
		catch (Exception $e)
		{
			return '';
		}

		if (!$post->load($id))
		{
			return '';
		}

		return 'index.php?option=' . $extension . '&view=' . $view . '&id=' . $post->ticket_id . '#p' . $id;
	}

	/**
	 * Method to index an item. The item must be a FinderIndexerResult object.
	 *
	 * @param   Result  $item  The item to index as an FinderIndexerResult object.
	 *
	 * @return  void
	 *
	 * @throws  Exception on database error.
	 * @since   5.0.0
	 */
	protected function index(Result $item)
	{
		if (!$item->public)
		{
			$this->remove($item->id);

			return;
		}

		// Check if the extension is enabled
		if (ComponentHelper::isEnabled($this->extension) == false)
		{
			return;
		}

		// Build the necessary route and path information.
		/** @var TicketTable $ticket */
		$ticket = $this->getFactory()->createTable('Ticket', 'Admnistrator');

		if (!$ticket->load($item->ticket_id))
		{
			return;
		}

		$item->url   = 'index.php?option=com_ats&view=ticket&id=' . $item->ticket_id . '&catid=' . $ticket->catid . '#p'
		               . $item->id;
		$item->route = $item->url;
		//$item->path  = $this->getContentPath($item->route);

		// Translate the state. Articles should only be published if the category is published.
		$item->state = $this->translateState($item->enabled, $item->cat_state);

		$item->summary = $item->body;

		// Add the type taxonomy data.
		$item->addTaxonomy('Type', 'Ticket');

		// Add the author taxonomy data.
		if (!empty($item->author))
		{
			$item->addTaxonomy('Author', $item->author);
		}

		// Add the category taxonomy data.
		$categories = Factory::getApplication()->bootComponent('com_ats')->getCategory(['published' => false, 'access' => false]);
		$category   = $categories->get($item->catid);

		// Category does not exist, stop here
		if (!$category)
		{
			return;
		}

		$item->addNestedTaxonomy(
			'Category', $category, $this->translateState($category->published), $category->access, $category->language
		);

		// Add the language taxonomy data.
		$item->addTaxonomy('Language', $item->language);

		// Get content extras.
		Helper::getContentExtras($item);

		// Index custom fields (Joomla 5.0 and later versions - the check is performed in the method).
		$this->addCustomFields($item, 'com_ats.ticket');

		// Index the item.
		$this->indexer->index($item);
	}

	/**
	 * Method to setup the indexer to be run.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   5.0.0
	 */
	protected function setup()
	{
		/**
		 * Debug mode in the CLI application results in out–of–memory condition because the debug query monitor uses up
		 * too much memory processing the barrage of queries we issue during indexing. Disabling the query monitor
		 * allows us to run a full index of thousands of tickets without hitting the memory limit. Evil, but necessary.
		 */
		if ($this->getApplication()->isClient('cli'))
		{
			method_exists($this, 'getDatabase') ? $this->getDatabase() : $this->db->setMonitor(null);
		}

		return true;
	}

	/**
	 * Use custom fields in the index and/or taxonomy
	 *
	 * @param   Result  $item     Result object to add the custom fields to
	 * @param   string  $context  Custom field context (should be 'com_ats.ticket')
	 *
	 * @return  void
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function addCustomFields(Result $item, string $context): void
	{
		// Do I have Joomla 5.0 or later?
		static $isSupported = null;

		$isSupported ??= version_compare(JVERSION, '5.0.0', 'ge');

		if (!$isSupported)
		{
			return;
		}

		// Get the custom fields
		$obj     = new \stdClass();
		$obj->id = $item->ticket_id ?? $item->id;

		$fields = FieldsHelper::getFields($context, $obj, true);

		// Make sure there are custom fields
		if (!is_array($fields) || empty($fields))
		{
			return;
		}

		// Filter out the fields which are not to be indexed
		$fields = array_filter($fields, fn($field) => $field->params->get('searchindex', 0) != 0);

		if (empty($fields))
		{
			return;
		}

		foreach ($fields as $field)
		{
			$searchIndex = $field->params->get('searchindex', 0);

			// Should this field be indexed?
			if (in_array($searchIndex, [1, 3]))
			{
				$name        = 'jsfield_' . $field->name;
				$item->$name = $field->value;
				$item->addInstruction(Indexer::META_CONTEXT, $name);
			}

			// Should this field be a taxonomy?
			if (in_array($searchIndex, [2, 3]) && !empty($field->value))
			{
				$item->addTaxonomy($field->title, $field->value, $field->state, $field->access, $field->language);
			}
		}
	}

	private function getFactory(): MVCFactoryInterface
	{
		if (!empty(self::$mvcFactory))
		{
			return self::$mvcFactory;
		}

		self::$mvcFactory = $this->getApplication()->bootComponent($this->extension)->getMVCFactory();

		return self::$mvcFactory;
	}

	/**
	 * Method to remove all posts of a ticket from the index
	 *
	 * @param   string  $id                The ID of the item to remove.
	 * @param   bool    $removeTaxonomies  Remove empty taxonomies
	 *
	 * @return  boolean  True on success.
	 *
	 * @throws  Exception on database error.
	 * @since   2.5
	 */
	private function removeAll($id, $removeTaxonomies = true)
	{
		// Get the item's URL
		$db  = method_exists($this, 'getDatabase') ? $this->getDatabase() : $this->db;
		$url = $db->quote(
			'index.php?option=' . $this->extension . '&view=' . $this->layout . '&id=' . $id .
			'#%'
		);

		// Get the link ids for the content items.
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select($db->quoteName('link_id'))
			->from($db->quoteName('#__finder_links'))
			->where($db->quoteName('url') . ' LIKE :url')
			->bind(':url', $url);
		$db->setQuery($query);
		$items = $db->loadColumn();

		// Check the items.
		if (empty($items))
		{
			$this->triggerPluginEvent('onFinderIndexAfterDelete', [$id]);

			return true;
		}

		// Remove the items.
		foreach ($items as $item)
		{
			$this->indexer->remove($item, $removeTaxonomies);
		}

		return true;
	}
}
