<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Plugin\User\ATS\Extension;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\AddonEmails;
use Akeeba\Component\ATS\Administrator\Table\PostTable;
use Akeeba\Component\ATS\Administrator\Table\TicketTable;
use Exception;
use Joomla\CMS\Form\Form;
use Joomla\CMS\MVC\Factory\MVCFactoryAwareTrait;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\DatabaseDriver;
use Joomla\Database\ParameterType;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;
use Joomla\Utilities\ArrayHelper;

class ATS extends CMSPlugin implements DatabaseAwareInterface, SubscriberInterface
{
	use DatabaseAwareTrait;
	use MVCFactoryAwareTrait;

	/** @inheritdoc */
	protected $autoloadLanguage = true;

	public static function getSubscribedEvents(): array
	{
		return [
			'onContentPrepareData' => 'onContentPrepareData',
			'onContentPrepareForm' => 'onContentPrepareForm',
			'onUserAfterDelete'    => 'onUserAfterDelete',
			'onUserAfterSave'      => 'onUserAfterSave',
		];
	}

	/**
	 * Load the ATS user profile values
	 *
	 * @param   Event  $event
	 *
	 * @return  void
	 * @since   1.0.0
	 */
	public function onContentPrepareData(Event $event): void
	{
		/**
		 * @var   string  $context  The context for the data
		 * @var   int     $data     The user id
		 */
		[$context, $data] = array_values($event->getArguments());
		$result = $event->getArgument('result', []);

		// Check we are manipulating a valid form.
		if (!in_array($context, ['com_users.profile', 'com_users.user', 'com_users.registration', 'com_admin.profile']))
		{
			$event->setArgument('result', array_merge($result, [true]));

			return;
		}

		/**
		 * The $data must be an object and have an id property but not a profile property (that'd be the profile already
		 * loaded by Joomla).
		 */
		if (!is_object($data) || !isset($data->id) || isset($data->ats))
		{
			$event->setArgument('result', array_merge($result, [true]));

			return;
		}

		// Get the user ID
		$userId = (int) ($data->id ?? 0);

		// Make sure we have a positive integer user ID
		if ($userId <= 0)
		{
			$event->setArgument('result', array_merge($result, [true]));

			return;
		}

		// Load the profile data from the database.
		try
		{
			$profileKey = 'ats.%';
			/** @var DatabaseDriver $db */
			$db    = $this->getDatabase();
			$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
				->select([
					$db->quoteName('profile_key'),
					$db->quoteName('profile_value'),
				])
				->from($db->quoteName('#__user_profiles'))
				->where($db->quoteName('user_id') . ' = :user_id')
				->where($db->quoteName('profile_key') . ' LIKE :profile_key')
				->order($db->quoteName('ordering'))
				->bind(':user_id', $userId, ParameterType::INTEGER)
				->bind(':profile_key', $profileKey, ParameterType::STRING);

			$results = $db->setQuery($query)->loadAssocList('profile_key', 'profile_value');

			$data->ats = [];
			foreach ($results as $k => $v)
			{
				$k = str_replace('ats.', '', $k);

				if ($k == 'notification_emails')
				{
					// Transparent data migration from older ATS versions
					$v = AddonEmails::encodeAddonEmailJSON(AddonEmails::decodeAddonEmailJSON($v));
				}

				$data->ats[$k] = $v;
			}
		}
		catch (Exception $e)
		{
			// We suppress any database error. It means we have no data set.
		}

		$event->setArgument('result', array_merge($result, [true]));
	}

	/**
	 * Inject the ATS user profile fields into the user form.
	 *
	 * @param   Event  $event
	 *
	 * @return  void
	 * @since   1.0.0
	 */
	public function onContentPrepareForm(Event $event): void
	{
		/**
		 * @var   Form   $form  The form to be altered.
		 * @var   array  $data  The associated data for the form.
		 */
		[$form, $data] = array_values($event->getArguments());
		$result = $event->getArgument('result', []);

		if (!($form instanceof Form))
		{
			$event->setArgument('result', array_merge($result, [true]));

			return;
		}

		// Check we are manipulating a valid form.
		if (!in_array($form->getName(), [
			'com_admin.profile', 'com_users.user', 'com_users.registration', 'com_users.profile',
		]))
		{
			$event->setArgument('result', array_merge($result, [true]));

			return;
		}

		// Add the registration fields to the form.
		Form::addFormPath(__DIR__ . '/../../ats');
		$form->loadFile('ats', false);

		if (!$this->params->get('show_signature', 1))
		{
			$form->removeField('signature', 'ats');
		}

		if (!$this->params->get('show_multiple_email', 0))
		{
			$form->removeField('notification_emails', 'ats');
		}

		$event->setArgument('result', array_merge($result, [true]));
	}

	/**
	 * Remove all user profile information for the given user ID
	 *
	 * Method is called after user data is deleted from the database
	 *
	 * @param   Event  $event
	 *
	 * @return  void
	 * @since   1.0.0
	 */
	public function onUserAfterDelete(Event $event): void
	{
		/**
		 * @var   array    $user     Holds the user data
		 * @var   boolean  $success  True if user was successfully stored in the database
		 * @var   string   $msg      Message
		 */
		[$user, $success, $msg] = array_values($event->getArguments());

		if (!$success)
		{
			return;
		}

		$userId = ArrayHelper::getValue($user, 'id', 0, 'int');

		if ($userId <= 0)
		{
			return;
		}

		/** @var DatabaseDriver $db */
		$db = $this->getDatabase();

		// Delete user profile information
		$profileKey = 'ats.%';
		$query      = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->delete($db->quoteName('#__user_profiles'))
			->where($db->quoteName('user_id') . ' = :user_id')
			->where($db->quoteName('profile_key') . ' LIKE :profile_key')
			->bind(':user_id', $userId, ParameterType::INTEGER)
			->bind(':profile_key', $profileKey, ParameterType::STRING);

		$db->setQuery($query)->execute();

		// Delete tickets
		$this->deleteTickets($userId);

		// Delete leftover posts
		$this->deletePosts($userId);
	}

	/**
	 * Saves user profile data
	 *
	 * @return  void
	 * @since   1.0.0
	 */
	public function onUserAfterSave(Event $event): void
	{
		/**
		 * @var   array   $data    Entered user data
		 * @var   bool    $isNew   TRUE if this is a new user
		 * @var   bool    $result  TRUE if saving the user worked
		 * @var   string  $error   Error message
		 */
		[$data, $isNew, $result, $error] = array_values($event->getArguments());

		$userId = ArrayHelper::getValue($data, 'id', 0, 'int');

		if (!$userId || !$result || !isset($data['ats']) || !count($data['ats']))
		{
			return;
		}

		/** @var DatabaseDriver $db */
		$db         = $this->getDatabase();
		$profileKey = 'ats.%';
		$query      = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->delete($db->quoteName('#__user_profiles'))
			->where($db->quoteName('user_id') . ' = :user_id')
			->where($db->quoteName('profile_key') . ' LIKE :profile_key')
			->bind(':user_id', $userId, ParameterType::INTEGER)
			->bind(':profile_key', $profileKey, ParameterType::STRING);;

		try
		{
			$db->setQuery($query)->execute();
		}
		catch (Exception $e)
		{
			return;
		}

		$order = 1;

		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->insert($db->quoteName('#__user_profiles'))
			->columns([
				$db->quoteName('user_id'),
				$db->quoteName('profile_key'),
				$db->quoteName('profile_value'),
				$db->quoteName('ordering'),
			]);

		foreach ($data['ats'] as $k => $v)
		{
			if ($k == 'notification_emails')
			{
				// The value may be an array of fields. In this case I need to process it differently.
				$isAssocEmails = !is_array($v) || array_reduce($v, function (bool $carry, $data) {
						return $carry && (empty($data) || !is_array($data));
					}, true);

				if (!$isAssocEmails)
				{
					$v = json_encode($v);
				}

				$emailAssoc = is_array($v) ? $v : AddonEmails::decodeAddonEmailJSON($v);
				$emailAssoc = AddonEmails::filterAddonEmails($emailAssoc, $userId);
				$v          = AddonEmails::encodeAddonEmailJSON($emailAssoc);
			}

			$query->values(implode(',', $query->bindArray([
				$userId,
				'ats.' . $k,
				$v,
				$order++,
			], [
				ParameterType::INTEGER,
				ParameterType::STRING,
				ParameterType::STRING,
				ParameterType::INTEGER,
			])));
		}

		$db->setQuery($query)->execute();
	}

	/**
	 * Deletes the leftover posts owned by a given user ID
	 *
	 * @param   int  $userId  The user ID which owns tickets to delete
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	private function deletePosts(int $userId): void
	{
		/** @var DatabaseDriver $db */
		$db    = $this->getDatabase();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select($db->quoteName('id'))
			->from($db->quoteName('#__ats_posts'))
			->where($db->quoteName('created_by') . ' = :created_by')
			->bind(':created_by', $userId);

		try
		{
			$postIds = $db->setQuery($query)->loadColumn() ?: [];
		}
		catch (Exception $e)
		{
			$postIds = [];
		}

		if (empty($postIds))
		{
			return;
		}

		/** @var PostTable $postTable */
		try
		{
			$postTable = $this->getMVCFactory()->createTable('Post', 'Administrator');
		}
		catch (Exception $e)
		{
			return;
		}

		array_walk($postIds, function ($id) use ($postTable) {
			try
			{
				$postTable->delete($id);
			}
			catch (Exception $e)
			{
				// No problem if it fails.
			}
		});
	}

	/**
	 * Deletes the tickets owned by a given user ID
	 *
	 * @param   int  $userId  The user ID which owns tickets to delete
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	private function deleteTickets(int $userId): void
	{
		/** @var DatabaseDriver $db */
		$db    = $this->getDatabase();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select($db->quoteName('id'))
			->from($db->quoteName('#__ats_tickets'))
			->where($db->quoteName('created_by') . ' = :created_by')
			->bind(':created_by', $userId);

		try
		{
			$ticketIds = $db->setQuery($query)->loadColumn() ?: [];
		}
		catch (Exception $e)
		{
			$ticketIds = [];
		}

		if (empty($ticketIds))
		{
			return;
		}

		/** @var TicketTable $ticketTable */
		try
		{
			$ticketTable = $this->getMVCFactory()->createTable('Ticket', 'Administrator');
		}
		catch (Exception $e)
		{
			return;
		}

		array_walk($ticketIds, function ($id) use ($ticketTable) {
			try
			{
				$ticketTable->delete($id);
			}
			catch (Exception $e)
			{
				// No problem if it fails.
			}
		});
	}
}
