<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Controller;

use Akeeba\Component\ATS\Administrator\Mixin\ControllerEventsTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerRegisterTasksTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerReturnURLTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerReusableModelsTrait;
use Akeeba\Component\ATS\Administrator\Model\ControlpanelModel;
use Akeeba\Component\ATS\Administrator\Model\UpdatesModel;
use Akeeba\Component\ATS\Administrator\Model\UpgradeModel;
use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Response\JsonResponse;

defined('_JEXEC') or die;

class ControlpanelController extends BaseController
{
	use ControllerEventsTrait;
	use ControllerReusableModelsTrait;
	use ControllerRegisterTasksTrait;
	use ControllerReturnURLTrait;

	/**
	 * Displays the CHANGELOG rendered as HTML
	 *
	 * @return  self
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function changelog(): self
	{
		$view = $this->getView();
		$view->setLayout('changelog');

		return $this->display(true);
	}

	/**
	 * Returns tickets and posts statistics per day.
	 *
	 * Only valid in a request with format=json. Needs anti-CSRF token.
	 *
	 * @since  5.0.0
	 */
	public function getTicketStats()
	{
		if (!$this->checkToken('get', false))
		{
			echo new JsonResponse(null, Text::_('JINVALID_TOKEN'), true);

			return;
		}

		$format = $this->input->get('format', 'html');

		if ($format != 'json')
		{
			throw new \RuntimeException(Text::_('COM_ATS_CONTROLPANEL_ERR_INVALIDFORMAT'), 500);
		}

		$defaultSince = (clone Factory::getDate())->sub(new \DateInterval('P1M'))->setTime(0, 0, 0, 0);
		$since        = $this->input->getString('since', $defaultSince->toISO8601());

		try
		{
			$earliestDate = clone Factory::getDate($since);
		}
		catch (Exception $e)
		{
			$earliestDate = $defaultSince;
		}

		$latestDate = (clone $earliestDate)->add(new \DateInterval('P1M'));

		// Create a default results array filled with zeroes.
		$results  = [];
		$thisDate = clone $earliestDate;

		while (true)
		{
			$key           = $thisDate->format('Y-m-d');
			$results[$key] = [
				'date'    => $key,
				'tickets' => 0,
				'posts'   => 0,
			];

			$thisDate->add(new \DateInterval('P1D'));

			if ($thisDate->getTimestamp() > $latestDate->getTimestamp())
			{
				break;
			}
		};

		/** @var ControlpanelModel $model */
		$model = $this->getModel();

		$from = clone $earliestDate;
		$to = clone $latestDate;

		foreach ($model->ticketsPerDay($from, $to) as $entry)
		{
			$results[$entry['date']]['tickets'] = $entry['tickets'];
		}

		$from = clone $earliestDate;
		$to = clone $latestDate;

		foreach ($model->postsPerDay($from, $to) as $entry)
		{
			$results[$entry['date']]['posts'] = $entry['posts'];
		}

		// IMPORTANT! array_values is necessary to reindex the array in a way I can more easily use with JavaScript.
		echo new JsonResponse(array_values($results));
	}

	/**
	 * Runs before main HTML view.
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function onBeforeMain()
	{
		$model = $this->getModel();

		$model->updateMagicParameters();

		/** @var UpdatesModel $updateModel */
		$updateModel = $this->getModel('Updates', 'Administrator');
		$updateModel->refreshUpdateSite();

		/** @var UpgradeModel $upgradeModel */
		$upgradeModel = $this->getModel('Upgrade', 'Administrator');
		$upgradeModel->init();
		$upgradeModel->adoptMyExtensions();

		$view = $this->getView();
		$view->setModel($this->getModel('Usagestatistics', 'Administrator'));
		$view->setModel($this->getModel('Updates', 'Administrator'));
	}
}