<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Controller;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\Permissions;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerEventsTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerRedirectionTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerNewPostTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerReturnURLTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerReusableModelsTrait;
use Akeeba\Component\ATS\Administrator\Model\PostModel;
use Akeeba\Component\ATS\Administrator\Table\PostTable;
use Akeeba\Component\ATS\Administrator\Table\TicketTable;
use Akeeba\Component\ATS\Administrator\View\Post\HtmlView;
use Exception;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Form\FormFactoryInterface;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\MVC\Controller\FormController;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\Input\Input;

class PostController extends FormController
{
	use ControllerEventsTrait;
	use ControllerReusableModelsTrait
	{
		ControllerReusableModelsTrait::getModel as reusableGetModel;
	}
	use ControllerReturnURLTrait
	{
		ControllerReturnURLTrait::getRedirectToListAppend as applyReturnURLOnListAppend;
	}
	use ControllerNewPostTrait;
	use ControllerRedirectionTrait;

	/** @inheritdoc */
	protected $text_prefix = 'COM_ATS_POST';

	/** @inheritdoc */
	protected $view_list = 'tickets';

	/** @inheritdoc */
	protected $name = 'post';

	/**
	 * Flag set by TicketController to indicate we are saving a guest ticket (no permissions checked)
	 *
	 * @var   bool
	 * @since 5.0.0
	 */
	private $guestTicket = false;

	/** @inheritdoc */
	public function __construct($config = [], ?MVCFactoryInterface $factory = null, ?CMSApplication $app = null, ?Input $input = null, ?FormFactoryInterface $formFactory = null)
	{
		$this->guestTicket = (bool) ($config['guestTicket'] ?? false);

		parent::__construct($config, $factory, $app, $input, $formFactory);
	}

	/** @inheritdoc */
	public function getModel($name = 'Post', $prefix = '', $config = [])
	{
		if ($this->guestTicket)
		{
			$config['guestTicket'] = true;
		}

		return $this->reusableGetModel($name, $prefix, $config);
	}

	public function edit($key = null, $urlVar = null)
	{
		// Joomla 4.1.1 and later will only allow cid as a POST variable. We need to use it with GET as well.
		$cid = (array) $this->input->get('cid', [], 'int');

		if (!empty($cid))
		{
			$this->input->post->set('cid', $cid);
		}

		return parent::edit($key, $urlVar);
	}

	/** @inheritdoc */
	protected function allowAdd($data = [])
	{
		$ticket_id = $data['ticket_id'] ?? null;

		if (empty($ticket_id))
		{
			return false;
		}

		/** @var TicketTable $ticket */
		$ticket = $this->getModel()->getTable('Ticket', 'Administrator');

		if (!$ticket->load($ticket_id))
		{
			return false;
		}

		if ($this->guestTicket)
		{
			return true;
		}

		$ticketCanDo = Permissions::getTicketPrivileges($ticket);
		$canDo       = Permissions::getTicketPrivileges($ticket);

		return $canDo['post'] && $ticketCanDo['view'];
	}

	/** @inheritdoc */
	protected function allowEdit($data = [], $key = 'id')
	{
		$ticket_id = $data['ticket_id'] ?? null;
		$post_id   = $data['id'] ?? null;

		if (empty($ticket_id) && empty($post_id))
		{
			return false;
		}

		/** @var PostTable $post */
		$post = $this->getModel()->getTable('Post', 'Administrator');
		/** @var TicketTable $ticket */
		$ticket = $this->getModel()->getTable('Ticket', 'Administrator');

		if (!empty($ticket_id))
		{
			if (!$ticket->load($ticket_id))
			{
				return false;
			}
		}

		if (!empty($post_id))
		{
			if (!$post->load($post_id))
			{
				return false;
			}
		}

		if (empty($ticket_id) && !empty($post_id))
		{
			$ticket = $post->getTicket();
		}

		if (empty($ticket->id))
		{
			return false;
		}

		$ticketCanDo = Permissions::getTicketPrivileges($ticket);
		$canDo       = Permissions::getPostPrivileges($post);

		return $canDo['edit'] && $ticketCanDo['view'];
	}

	/**
	 * Runs after the save task (Save & Close button).
	 *
	 * Used to apply the return URL.
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	protected function onAfterSave(): void
	{
		$this->applyReturnUrl();
	}

	/**
	 * Runs before the main display task.
	 *
	 * Used to communicate the return URL to the view.
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	protected function onBeforeMain(): void
	{
		/** @var HtmlView $view */
		$view            = $this->getView();
		$view->returnUrl = $this->getReturnUrl();
	}

	/**
	 * Called after we have successfully handled saving a post.
	 *
	 * Used to handle attachments, user tags and post notifications (e.g. emails).
	 *
	 * @param   BaseDatabaseModel|PostModel  $model      The model this controller acts upon
	 * @param   array                        $validData  Validated data being saved
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	protected function postSaveHook(BaseDatabaseModel $model, $validData = [])
	{
		// Make sure we have a post where we are attaching the uploads
		$postId = $model->getState('post.id', $validData['id'] ?? null);

		if (!$postId)
		{
			return;
		}

		/** @var PostTable $post */
		$post  = $model->getItem();
		$canDo = Permissions::getPostPrivileges($post);

		// Handle attachments if allowed
		if ($canDo['attachment'])
		{
			$this->handleAttachments($post);
		}

		// Handle user tags if allowed
		if ($canDo['admin'])
		{
			$this->handleUserTags($post);
		}

		// Send post notifications (emails etc)
		$newPost = $model->getState('post.isNew', true);
		$model->setState('post.isNew', null);
		$this->postNotifiable($post, $newPost);
	}

	protected function getRedirectToListAppend()
	{
		if (!$this->app->isClient('site'))
		{
			return parent::getRedirectToListAppend();
		}

		$returnUrl = $this->getReturnUrl();

		if ($returnUrl)
		{
			$this->input->set('return', base64_encode($returnUrl));

			return '';
		}

		/** @var PostModel $model */
		$model = $this->getModel();

		try
		{
			$isNew  = $model->getState('post.isNew', false);
			$post   = $model->getItem();
			$ticket = $post->getTicket();
		}
		catch (Exception $e)
		{
			$isNew = false;
		}

		if (!$isNew || !$post->getId() || !$ticket->getId())
		{
			return parent::getRedirectToListAppend();
		}

		$urlParams = [
			'option'   => 'com_ats',
			'view'     => 'ticket',
			'layout'   => 'default',
			'id'       => $ticket->getId(),
			'catid'    => $ticket->catid,
			'format'   => 'html',
			'Itemid'   => $this->input->getInt('Itemid', null),
			'language' => Multilanguage::isEnabled() ? $this->app->getLanguage()->getTag() : null,
		];

		$urlParams = array_filter($urlParams, function ($x) {
			return !is_null($x);
		});

		$this->input->set('return', base64_encode('index.php?' . http_build_query($urlParams)));

		return '';
	}
}