<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Controller;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\Permissions;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerEventsTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerNewPostTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerReturnURLTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerReusableModelsTrait;
use Akeeba\Component\ATS\Administrator\Model\TicketModel;
use Akeeba\Component\ATS\Administrator\Table\PostTable;
use Akeeba\Component\ATS\Administrator\Table\TicketTable;
use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\FormController;
use Joomla\CMS\Router\Route;

class TicketController extends FormController
{
	use ControllerEventsTrait;
	use ControllerReusableModelsTrait;
	use ControllerNewPostTrait;
    use ControllerReturnURLTrait;

	/** @inheritdoc */
	protected $text_prefix = 'COM_ATS_TICKET';

	/** @inheritdoc */
	public function batch($model = null)
	{
		$this->checkToken();

		/** @var TicketModel $model */
		$model = $this->getModel('Ticket', 'Administrator', []);

		// Preset the redirect
		$this->setRedirect(Route::_('index.php?option=com_ats&view=tickets' . $this->getRedirectToListAppend(), false));

		return parent::batch($model);
	}

	public function save($key = null, $urlVar = null)
	{
		// Token check
		$this->checkToken();

		// Get the jform data and determine if it's a new ticket
		$data = $this->input->post->get('jform', [], 'array');
		$id   = $data['id'] ?? $this->input->getInt('id', null);

		// Existing ticket. Just use the default Joomla behaviour and be done with it.
		if (is_numeric($id) && ($id > 0))
		{
			$return = parent::save($key, $urlVar);

			/**
			 * If this is the frontend of the site, the save succeeded without changing the ticket ID, this it the
			 * ticket owner who's changing the ticket and the ticket is not closed we will also send an automatic post
			 * to notify the support staff that the client has edited the ticket.
			 */

			/** @var TicketModel $model */
			$model = $this->getModel();

			if (!$return || !$this->app->isClient('site') || ($model->getState('ticket.id') != $id))
			{
				return $return;
			}

			/** @var TicketTable $ticket */
			$ticket = $model->getTable('ticket', 'administrator');

			if (!$ticket->load($id))
			{
				return $return;
			}

			$me = Permissions::getUser();

			if ((!$me->guest && ($me->id == $ticket->created_by)) && ($ticket->status != 'C'))
			{
				$this->sendTicketModifiedPost($ticket);
			}

			return $return;
		}

		// OK. We're here. That's a new ticket. Split the data for the ticket and the post.
		/** @var TicketModel $model */
		$model    = $this->getModel();
		$form     = $model->getForm($data, true);
		$fields   = $form->getFieldset('post');
		$postKeys = array_map(function (FormField $field) {
			return $field->fieldname;
		}, $fields);
		$postData = array_filter($data, function ($key) use ($postKeys) {
			return in_array($key, $postKeys);
		}, ARRAY_FILTER_USE_KEY);
		$data     = array_filter($data, function ($key) use ($postKeys) {
			return !in_array($key, $postKeys);
		}, ARRAY_FILTER_USE_KEY);

		// Set jform in the request
		$this->input->set('jform', $data);
		$this->input->post->set('jform', $data);

		// Call the parent method. Setting ticket.savingNew modifies the form to exclude the post data.
		$model->setState('ticket.savingNew', true);

		$result = parent::save($key, $urlVar);

		$model->setState('ticket.savingNew', null);

		// If the ticket creation fails we will not handle the post creation at all.
		if (!$result)
		{
			$context = "$this->option.edit.$this->context";
			$this->app->setUserState($context . '.data', array_merge($data, $postData));

			// We need to create a different return URL which takes us back to where we were before.
			$urlParams = [
				'option'   => 'com_ats',
				'view'     => 'ticket',
				'layout'   => 'newticket',
				'id'       => '',
				'catid'    => $form->getValue('catid'),
				'format'   => 'html',
				'Itemid'   => $this->input->getInt('Itemid', null),
				'language' => Multilanguage::isEnabled() ? $this->app->getLanguage()->getTag() : null,
				'return'   => $this->input->get('return', null, 'base64'),
			];

			$urlParams = array_filter($urlParams, function ($x) {
				return !is_null($x);
			});

			$this->input->set('return', base64_encode('index.php?' . http_build_query($urlParams)));

			return false;
		}

		// Update the post data with the newly created ticket's ID
		$savedTicket = $model->getItem();

		$expandedPostData = array_merge($postData, [
			'ticket_id' => $savedTicket->id,
		]);
		$this->input->set('jform', $expandedPostData);
		$this->input->post->set('jform', $expandedPostData);

		// Call the save() method on PostController
		$prefix = $this->app->isClient('site') ? 'Site' : 'Administrator';
		/** @var PostController $postController */
		$postController = $this->factory->createController('Post', $prefix, [
			'guestTicket' => $model->getState('guestTicket', false),
		], $this->app, $this->input);

		$result = $postController->execute('save');

		// All good? Return true.
		if ($result)
		{
			return true;
		}

		/**
		 * If saving the post failed I need to:
		 * - delete the ticket I created
		 * - set the session data again (it's already cleared by parent::save()...)
		 * - redirect to the new ticket page
		 */

		$savedTicket->delete();

		$context = "$this->option.edit.$this->context";
		$this->app->setUserState($context . '.data', array_merge($data, $postData));

		// We need to create a different return URL which takes us back to where we were before.
		$urlParams = [
			'option'   => 'com_ats',
			'view'     => 'ticket',
			'layout'   => 'edit',
			'id'       => '',
			'catid'    => $form->getValue('catid'),
			'format'   => 'html',
			'Itemid'   => $this->input->getInt('Itemid', null),
			'language' => Multilanguage::isEnabled() ? $this->app->getLanguage()->getTag() : null,
			'return'   => $this->input->get('return', null, 'base64'),
		];

		$urlParams = array_filter($urlParams, function ($x) {
			return !is_null($x);
		});

		$this->input->set('return', base64_encode('index.php?' . http_build_query($urlParams)));

		// Pass the message from the Post controller
		$this->setMessage($postController->getRedirectionMessage(), $postController->getRedirectionMessageType());

		return false;
	}

    public function invite()
    {
        $ticket_id   = $this->input->getInt('id');
        $invite_user = $this->input->getUsername('invite_user', '');

        /** @var TicketModel $model */
        $model = $this->getModel();
        $msg   = Text::_('COM_ATS_TICKET_INVITE_SUCCESS');
        $type  = 'info';

        try
        {
            $model->inviteUser($ticket_id, $invite_user);

            $date = clone Factory::getDate();

            // Send a “System User” post
            $post = [
                'ticket_id'    => $ticket_id,
                'content_html' => Text::sprintf('COM_ATS_TICKET_INVITE_USER_ADDED', $invite_user),
                'created'      => $date->toSql(),
                'created_by'   => -1,
                'enabled'      => 1,
            ];

            /** @var PostTable $postTable */
            $postTable = $this->factory->createTable('post', 'administrator');

            $postTable->setUpdateCreated(false);
            $postTable->setUpdateModified(false);
            $postTable->save($post);
            $postTable->setUpdateCreated(true);
            $postTable->setUpdateModified(true);

            $this->postNotifiable($postTable, true);
        }
        catch (Exception $e)
        {
            $msg  = $e->getMessage();
            $type = 'notice';
        }

        $app = Factory::getApplication();
        $app->enqueueMessage($msg, $type);

        $this->applyReturnUrl();
    }

	/** @inheritdoc */
	protected function allowEdit($data = [], $key = 'catid')
	{
		$id         = $data['id'] ?? null;
		$categoryId = $data[$key] ?? 0;

		/** @var TicketTable $ticket */
		$ticket = $this->getModel()->getTable();

		if ($ticket->load($id))
		{
			$permissions = Permissions::getTicketPrivileges($ticket);

			return $permissions['edit'];
		}

		if ($categoryId)
		{
			return $this->app->getIdentity()->authorise('core.edit', $this->option . '.category.' . $categoryId)
				|| $this->app->getIdentity()->authorise('core.edit', $this->option);
		}

		return parent::allowEdit();
	}

	protected function onBeforeMain()
	{
		$view        = $this->getView();
		$ticketModel = $this->getModel();
		$view->setModel($ticketModel, true);
		$view->setModel($this->getModel('Post', '', ['ignore_request' => true]), false);

		if (defined('ATS_PRO') && ATS_PRO)
		{
			$view->setModel($this->getModel('Managernote', '', ['ignore_request' => true]), false);
		}

		$jForm = $this->input->get('jform', []);
		$catId = $this->input->getInt('catid', $jForm['catid'] ?? 0);
		$ticketModel->setState('ticket.catid', $catId ?: null);
	}

	/**
	 * Send an automatic post when the ticket information is changed by the ticket owner
	 *
	 * @param   TicketTable  $ticket  The ticket which has been edited
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function sendTicketModifiedPost(TicketTable $ticket): void
	{
		// Get some basic information
		$me   = Permissions::getUser();
		$date = clone Factory::getDate();

		// Send a “System User” post
		$post = [
			'ticket_id'    => $ticket->getId(),
			'content_html' => Text::sprintf('COM_ATS_TICKET_HAS_BEEN_EDITED', $me->name, $me->username),
			'created'      => $date->toSql(),
			'created_by'   => -1,
			'enabled'      => 1,
		];

		/** @var PostTable $postTable */
		$postTable = $this->factory->createTable('post', 'administrator');

		$postTable->setUpdateCreated(false);
		$postTable->setUpdateModified(false);
		$postTable->save($post);
		$postTable->setUpdateCreated(true);
		$postTable->setUpdateModified(true);

		$this->postNotifiable($postTable, true);

		// Update the ticket information.
		$ticket->status      = 'O';
		$ticket->modified_by = $me->id;
		$ticket->modified    = $date->toSql();

		TicketTable::$noPermissionsCheck = 1;

		$ticket->store();

		TicketTable::$noPermissionsCheck = 0;
	}
}