<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Controller;

use Akeeba\Component\ATS\Administrator\Helper\Permissions;
use Akeeba\Component\ATS\Administrator\Mixin\CMSObjectWorkaroundTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerEventsTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerReturnURLTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ControllerReusableModelsTrait;
use Akeeba\Component\ATS\Administrator\Model\TicketModel;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\AdminController;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\Input\Input;
use Joomla\Utilities\ArrayHelper;

defined('_JEXEC') or die;

class TicketsController extends AdminController
{
	use ControllerEventsTrait;
	use ControllerReusableModelsTrait;
	use ControllerReturnURLTrait;
	use CMSObjectWorkaroundTrait;

	/** @inheritdoc  */
	protected $text_prefix = 'COM_ATS_TICKETS';

	/** @inheritdoc  */
	public function __construct($config = [], ?MVCFactoryInterface $factory = null, ?CMSApplication $app = null, ?Input $input = null)
	{
		parent::__construct($config, $factory, $app, $input);

		$this->registerTask('makeprivate', 'makepublic');
		$this->registerTask('reopen', 'close');
	}

	/** @inheritdoc  */
	public function getModel($name = 'Ticket', $prefix = 'Administrator', $config = ['ignore_request' => true])
	{
		return parent::getModel($name, $prefix, $config);
	}

	/**
	 * Method to make public (or private) a list of items
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	public function makepublic()
	{
		// Check for request forgeries
		$this->checkToken();

		// Get items to publish from the request.
		$cid   = $this->input->get('cid', [], 'array');
		$data  = ['makepublic' => 1, 'makeprivate' => 0];
		$task  = $this->getTask();
		$value = ArrayHelper::getValue($data, $task, 0, 'int');

		if (empty($cid))
		{
			$this->app->getLogger()->warning(Text::_($this->text_prefix . '_NO_ITEM_SELECTED'), ['category' => 'jerror']);
		}
		else
		{
			// Get the model.
			/** @var TicketModel $model */
			$model = $this->getModel();

			// Make sure the item ids are integers
			$cid = ArrayHelper::toInteger($cid);

			// Publish the items.
			try
			{
				[$dummy1, $dummy2, $errors] = $this->cmsObjectSafeCall($model, 'makepublic', $cid, $value);
				$ntext  = null;

				if ($value === 1)
				{
					if ($errors)
					{
						Factory::getApplication()->enqueueMessage(Text::plural($this->text_prefix . '_N_ITEMS_FAILED_MAKEPUBLIC', \count($cid)), 'error');
					}
					else
					{
						$ntext = $this->text_prefix . '_N_ITEMS_MADEPUBLIC';
					}
				}
				elseif ($value === 0)
				{
					$ntext = $this->text_prefix . '_N_ITEMS_MADEPRIVATE';
				}

				if (\count($cid))
				{
					$this->setMessage(Text::plural($ntext, \count($cid)));
				}
			}
			catch (\Exception $e)
			{
				$this->setMessage($e->getMessage(), 'error');
			}
		}

		$this->setRedirect(
			Route::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_list
				. $this->getRedirectToListAppend(), false
			)
		);
	}

	/**
	 * Method to make close and reopen a number of tickets
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	public function close()
	{
		// Check for request forgeries
		$this->checkToken();

		// Get items to publish from the request.
		$cid   = $this->input->get('cid', [], 'array');
		$data  = ['close' => 'C', 'reopen' => 'O'];
		$task  = $this->getTask();
		$value = ArrayHelper::getValue($data, $task, 'C', 'string');

		if (empty($cid))
		{
			$this->app->getLogger()->warning(Text::_($this->text_prefix . '_NO_ITEM_SELECTED'), ['category' => 'jerror']);
		}
		else
		{
			// Get the model.
			/** @var TicketModel $model */
			$model = $this->getModel();

			// Make sure the item ids are integers
			$cid = ArrayHelper::toInteger($cid);

			// Publish the items.
			try
			{
				[$dummy1, $dummy2, $errors] = $this->cmsObjectSafeCall($model, 'closeOpenTicket', $cid, $value);
				$ntext  = null;

				if ($value === 'C')
				{
					if ($errors)
					{
						var_dump($errors);
						Factory::getApplication()->enqueueMessage(Text::plural($this->text_prefix . '_N_ITEMS_FAILED_CLOSE', \count($cid)), 'error');
					}
					else
					{
						$ntext = $this->text_prefix . '_N_ITEMS_CLOSED';
					}
				}
				elseif ($value === 'O')
				{
					$ntext = $this->text_prefix . '_N_ITEMS_OPENED';
				}

				if (\count($cid))
				{
					$this->setMessage(Text::plural($ntext, \count($cid)));
				}
			}
			catch (\Exception $e)
			{
				$this->setMessage($e->getMessage(), 'error');
			}
		}

		$this->setRedirect(
			Route::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_list
				. $this->getRedirectToListAppend(), false
			)
		);
	}

    public function removeinvite()
    {
        // Check for request forgeries
        $this->checkToken();

        $cid          = $this->input->getInt('cid', 0);
        $invited_user = $this->input->getInt('userid');

        try
        {
            /** @var TicketModel $model */
            $model = $this->getModel();
            $model->removeInvite($invited_user, $cid);

            $this->setMessage(Text::_($this->text_prefix . '_INVITE_REMOVED'));
        }
        catch (\Exception $e)
        {
            $this->setMessage($e->getMessage(), 'error');
        }

        $redirect = Route::_('index.php?option=' . $this->option . '&view=' . $this->view_list . $this->getRedirectToListAppend(), false);

        if ($return_url = $this->input->getBase64('returnurl'))
        {
            $redirect = base64_decode($return_url);
        }

	    // If I am removing myself I should be redirected to the site's root to avoid seeing an "Access Denied" error.
	    if (
			$this->app->isClient('site')
			&& $invited_user === Permissions::getUser()->id
			&& !empty($invited_user)
	    )
	    {
		    $this->input->set('return', null);
		    $this->input->set('returnurl', null);

		    $redirect = Uri::base();
	    }

	    $this->setRedirect($redirect);
    }

	public function checkToken($method = 'request', $redirect = true)
	{
		return parent::checkToken($method, $redirect);
	}
}