<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Mixin;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\Attachment;
use Akeeba\Component\ATS\Administrator\Helper\UserTags;
use Akeeba\Component\ATS\Administrator\Model\EmailSending;
use Akeeba\Component\ATS\Administrator\Table\ManagernoteTable;
use Akeeba\Component\ATS\Administrator\Table\PostTable;
use Exception;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Event\Event;
use RuntimeException;

/**
 * Utility functions for handling new Posts
 *
 * @since  5.0.0
 */
trait ControllerNewPostTrait
{
	/**
	 * Handle any attachments in the request. Assumes $this is a Controller.
	 *
	 * This method finishes the upload of the attachments, creates AttachmentTable records, assigns them to the post and
	 * updates the post itself with the new list of attachments.
	 *
	 * @param   PostTable  $post
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	private function handleAttachments(PostTable $post): void
	{
		if (!defined('ATS_PRO') || !ATS_PRO)
		{
			return;
		}

		/**
		 * Get the jform input array from the $_FILES superglobal in a safe manner.
		 *
		 * This is just the file upload data, not the entire submitted form. The rest of the form data is in the $_POST
		 * superglobal accessed through $this->app->getInput()->post.
		 */
		$jFormFiles = $this->input->files->get('jform');

		// Make sure there are uploaded files we need to process!
		if (!isset($jFormFiles['attachments']) || empty($jFormFiles['attachments']))
		{
			return;
		}

		/**
		 * Array reshape.
		 *
		 * jform[attachments] is of the shape ['_someRandomId' => ['file' => [...]],
		 * '_someRandomId2' => ['file' => [...]], ...]. For processing purposes we only need the values of the 'file'
		 * keys in the inner arrays.
		 */
		$attachments = $jFormFiles['attachments'];
		$attachments = array_map(function ($x) {
			return $x['file'];
		}, $attachments);

		$result = Attachment::manageUploads($attachments, $post);
		/**
		 * @var  string $attachment_id
		 * @var  array  $errors
		 */
		extract($result);

		foreach ($errors as $error)
		{
			$this->app->enqueueMessage($error, CMSApplication::MSG_WARNING);
		}

		// Update the post with the attachment IDs.
		if (!empty($attachment_id))
		{
			$post->save([
				'attachment_id' => $attachment_id,
			]);
		}
	}

	/**
	 * Apply the user tags submitted with a Post. Assumes $this is a Controller.
	 *
	 * @param   PostTable  $post  The post which was submitted.
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function handleUserTags(PostTable $post)
	{
		if (!defined('ATS_PRO') || !ATS_PRO)
		{
			return;
		}

		$jForm = $this->input->get('jform', []);

		if (!is_array($jForm))
		{
			return;
		}

		$userId = $post->getTicket()->created_by;

		if (!isset($jForm['usertags']))
		{
			// This actually happens when we try to remove all tags.
			UserTags::removeUserTags($userId);

			return;
		}

		if (defined('ATS_PRO') && ATS_PRO)
		{
			UserTags::setUserTags($userId, $jForm['usertags']);
		}
	}

	/**
	 * Handles notifications when saving a post.
	 *
	 * Can be used standalone, even in plugins.
	 *
	 * @param   PostTable  $post     The post to send notifications for. NULL for item returned by getItem().
	 * @param   bool       $newPost  Is this a new post? NULL to use post.isNew state variable.
	 *
	 * @return  void
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function postNotifiable(PostTable $post, bool $newPost): void
	{
		// Reference to the TicketTable object
		try
		{
			$ticket = $post->getTicket();
		}
		catch (RuntimeException $e)
		{
			return;
		}

		// Send emails, if necessary and possible
		if (class_exists(EmailSending::class) && (ComponentHelper::getParams('com_ats')->get('sendEmails', 1) == 1))
		{
			(new EmailSending())->sendPostEmails($post, $ticket, $newPost);
		}

		// Trigger plugin events
		PluginHelper::importPlugin('ats');

		$event = new Event('onATSPost', [
			'new'    => $newPost,
			'post'   => clone $post,
			'ticket' => $ticket,
		]);

		$post->getDispatcher()->dispatch('onATSPost', $event);
	}

	private function managerNoteNotifiable(ManagernoteTable $note): void
	{
		try
		{
			$ticket = $note->getTicket();
		}
		catch (RuntimeException $e)
		{
			return;
		}

		// Send emails, if necessary and possible
		if (class_exists(EmailSending::class) && (ComponentHelper::getParams('com_ats')->get('sendEmails', 1) == 1))
		{
			(new EmailSending())->sendNoteEmails($note, $ticket);
		}

		// Trigger plugin events
		PluginHelper::importPlugin('ats');

		$event = new Event('onATSManagerNote', [
			'note'   => clone $note,
			'ticket' => $ticket,
		]);

		$note->getDispatcher()->dispatch('onATSPost', $event);
	}
}