<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Model;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Mixin\ComponentObjectTrait;
use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Date\Date;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\Uri\Uri;
use Joomla\Database\ParameterType;

#[\AllowDynamicProperties]
class ControlpanelModel extends BaseDatabaseModel
{
	use ComponentObjectTrait;

	/**
	 * Counts tickets of a particular visibility and status for stats display
	 *
	 * @param   bool   $public  Count public (true) or private (false) tickets?
	 * @param   array  $status  State of the tickets to publish: array of values O, P, C, 1 ... 99
	 *
	 * @return  int  Number of tickets matchign this criteria
	 *
	 * @since   5.0.0
	 */
	public function countTickets(bool $public, array $status): int
	{
		$publicFlag = $public ? 1 : 0;

		$db    = $this->getDatabase();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('COUNT(*)')
			->from($db->quoteName('#__ats_tickets'))
			->where($db->quoteName('public') . ' = :public')
			->whereIn($db->quoteName('status'), $status, ParameterType::STRING)
			->bind(':public', $publicFlag, ParameterType::INTEGER);

		try
		{
			return $db->setQuery($query)->loadResult() ?: 0;
		}
		catch (Exception $e)
		{
			return 0;
		}
	}

	/**
	 * Does the user need to enter a Download ID in the component's Options page?
	 *
	 * @return  bool
	 * @throws  Exception
	 * @since   5.0.0
	 */
	public function needsDownloadID(): bool
	{
		// Do I need a Download ID?
		if (!ATS_PRO)
		{
			return false;
		}

		/** @var UpdatesModel $updateModel */
		$updateModel = $this->getMVCFactory()->createModel('Updates', 'Administrator', ['ignore_request' => true]);
		$dlid        = $updateModel->sanitizeLicenseKey($updateModel->getLicenseKey());

		return !$updateModel->isValidLicenseKey($dlid);
	}

	/**
	 * Returns the number of new posts submitted per day for the given period of time.
	 *
	 * @param   Date  $since  Earliest date and time to look for posts
	 * @param   Date  $to     Latest date and time to look for posts
	 *
	 * @return  array  Each item has two keys: date and posts
	 *
	 * @since   5.0.0
	 */
	public function postsPerDay(Date $since, Date $to): array
	{
		$db       = $this->getDatabase();
		$sinceSql = $since->toSql($db);
		$toSql    = $to->toSql($db);
		$query    = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select([
				'DATE(' . $db->quoteName('created') . ') AS ' . $db->quoteName('date'),
				'COUNT(' . $db->quoteName('id') . ') AS ' . $db->quoteName('posts'),
			])
			->from($db->quoteName('#__ats_posts'))
			->where($db->quoteName('created') . ' BETWEEN :since AND :to')
			->group($db->quoteName('date'))
			->order($db->quoteName('date') . 'ASC')
			->bind(':since', $sinceSql, ParameterType::STRING)
			->bind(':to', $toSql, ParameterType::STRING);

		try
		{
			return $db->setQuery($query)->loadAssocList() ?: [];
		}
		catch (Exception $e)
		{
			return [];
		}
	}

	/**
	 * Returns the number of new tickets submitted per day for the given period of time.
	 *
	 * @param   Date  $since  Earliest date and time to look for tickets
	 * @param   Date  $to     Latest date and time to look for tickets
	 *
	 * @return  array  Each item has two keys: date and tickets
	 *
	 * @since   5.0.0
	 */
	public function ticketsPerDay(Date $since, Date $to): array
	{
		$db       = $this->getDatabase();
		$sinceSql = $since->toSql($db);
		$toSql    = $to->toSql($db);
		$query    = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select([
				'DATE(' . $db->quoteName('created') . ') AS ' . $db->quoteName('date'),
				'COUNT(' . $db->quoteName('id') . ') AS ' . $db->quoteName('tickets'),
			])
			->from($db->quoteName('#__ats_tickets'))
			->where($db->quoteName('created') . ' BETWEEN :since AND :to')
			->group($db->quoteName('date'))
			->order($db->quoteName('date') . 'ASC')
			->bind(':since', $sinceSql, ParameterType::STRING)
			->bind(':to', $toSql, ParameterType::STRING);

		try
		{
			return $db->setQuery($query)->loadAssocList() ?: [];
		}
		catch (Exception $e)
		{
			return [];
		}
	}

	/**
	 * Update the cached live site's URL for the front-end scheduling feature.
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	public function updateMagicParameters()
	{
		$cParams = ComponentHelper::getParams($this->option);

		$cParams->set('siteurl', Uri::root());

		$this->getExtensionObject()
			->getComponentParametersService()
			->save($cParams);
	}
}