<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Model;

use Akeeba\Component\ATS\Administrator\Helper\AddonEmails;
use Akeeba\Component\ATS\Administrator\Helper\Permissions;
use Akeeba\Component\ATS\Administrator\Helper\TemplateEmails;
use Akeeba\Component\ATS\Administrator\Helper\URLTransformation;
use Akeeba\Component\ATS\Administrator\Table\ManagernoteTable;
use Akeeba\Component\ATS\Administrator\Table\PostTable;
use Akeeba\Component\ATS\Administrator\Table\TicketTable;
use Akeeba\Component\ATS\Site\Service\Category;
use Exception;
use Joomla\CMS\Access\Access;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Mail\Mail;
use Joomla\CMS\Mail\MailerFactoryInterface;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\User\User;
use Joomla\Database\ParameterType;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;

defined('_JEXEC') or die;

/**
 * Handles sending emails for new and modified posts, new tickets and ticket assignments.
 *
 * @since  5.0.0
 */
#[\AllowDynamicProperties]
class EmailSending extends BaseDatabaseModel
{
	/**
	 * Sends emails when a ticket is assigned to a different member of the support staff
	 *
	 * @param   TicketTable  $ticket  The ticket the post belongs to
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	public function sendAssignedEmails(TicketTable $ticket): void
	{
		// Is the template email helper available?
		if (!class_exists(TemplateEmails::class))
		{
			return;
		}

		if (empty($ticket->assigned_to))
		{
			return;
		}

		// Set up the mailer and the email template data
		$emailData = $this->getTicketEmailData($ticket);
		$mailer    = $this->getMailer($ticket);

		// Send manager emails
		$emailKey = 'com_ats.manager_assignedticket';
		$user     = Permissions::getUser($ticket->assigned_to);

		$this->applyAddonEmails($user, $mailer);

		TemplateEmails::sendMail($emailKey, array_merge($emailData, [
			'user_name'     => $user->name,
			'user_email'    => $user->email,
			'user_username' => $user->username,
		]), $user, null, false, $mailer);
	}

	/**
	 * Send emails about manager notes
	 *
	 * @param   ManagernoteTable  $note    The manager's note
	 * @param   TicketTable       $ticket  The ticket the note belongs to
	 *
	 *
	 * @throws \PHPMailer\PHPMailer\Exception
	 * @since  5.0.0
	 */
	public function sendNoteEmails(ManagernoteTable $note, TicketTable $ticket): void
	{
		// Is the template email helper available?
		if (!class_exists(TemplateEmails::class))
		{
			return;
		}

		// No emails are sent for modified manager notes
		if (!empty($note->modified) && ($note->modified !== $note->created) && ($note->modified_by ?? 0) !== $note->created_by)
		{
			return;
		}

		// Set up the mailer and the email template data
		$emailData = $this->getNoteEmailData($note, $ticket);
		$mailer    = $this->getMailer($ticket);

		// By default, send manager emails to all managers
		$managers        = array_filter(Permissions::getManagers($ticket->catid), function ($info) use ($note) {
			return $info->id != $note->created_by;
		});
		$recipients      = array_map(function ($info) {
			return Permissions::getUser($info->id);
		}, $managers);

		// Do I need to add “always notify” managers? This replaces the list of this category's managers.
		$recipients = $this->applyNotifyManagers($ticket, $recipients);

		// Skip not assigned managers
		$skipNotAssigned = ComponentHelper::getParams('com_ats')->get('assigned_noemail', 0) == 1;

		if ($skipNotAssigned && ($ticket->assigned_to ?? 0) > 0)
		{
			$recipients = array_filter($recipients, function (User $user) use ($ticket) {
				return $user->id == $ticket->assigned_to;
			});
		}

		// Do I need to remove managers? Removing managers overrides everything else, including explicitly added managers.
		$recipients = $this->applyExcludeManagers($ticket, $recipients);

		// Deduplicate the recipients
		$recipients = $this->deduplicateRecipients($recipients);

		// Send the emails
		foreach ($recipients as $user)
		{
			$thisMailer = clone $mailer;

			$this->applyAddonEmails($user, $thisMailer);

			TemplateEmails::sendMail('com_ats.managernote_new', array_merge($emailData, [
				'user_name'     => $user->name,
				'user_email'    => $user->email,
				'user_username' => $user->username,
			]), $user, null, false, $thisMailer);
		}
	}

	/**
	 * Sends emails when a post is created or modified
	 *
	 * @param   PostTable    $post     The created or modified post
	 * @param   TicketTable  $ticket   The ticket the post belongs to
	 * @param   bool         $newPost  Is this a new post?
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	public function sendPostEmails(PostTable $post, TicketTable $ticket, bool $newPost): void
	{
		// Is the template email helper available?
		if (!class_exists(TemplateEmails::class))
		{
			return;
		}

		// Get some basic information about the ticket and the post
		$modifiedByOwner = !$newPost && ($post->modified_by >= 0) && ($post->modified_by == $ticket->created_by);
		$createdByOwner  = $post->created_by == $ticket->created_by;

		// Is this a new ticket?
		try
		{
			$isNewTicket = $newPost && ($this->getFirstPost($ticket)->id == $post->id);
		}
		catch (Exception $e)
		{
			$isNewTicket = false;
		}

		// Set up the mailer and the email template data
		$emailData = $this->getPostEmailData($post, $ticket);
		$mailer    = $this->getMailer($ticket);

		$this->attachFiles($post, $mailer, $emailData);

		// Initial set up for manager emails
		$ticketVisibilityForEmailKey = $ticket->public ? 'public' : 'private';
		$suffixForEmailKey           = $isNewTicket ? 'new' : 'old';

		$emailKey   =
			$newPost
			? sprintf("com_ats.manager_%s_%s", $ticketVisibilityForEmailKey, $suffixForEmailKey)
			: 'com_ats.manager_edit';
		$recipients = array_map(function ($info) {
			return Permissions::getUser($info->id);
		}, Permissions::getManagers($ticket->catid));

		// Should I skip emailing non-assigned managers?
		if (
			(ComponentHelper::getParams('com_ats')->get('assigned_noemail', 0) == 1)
			&& ($ticket->assigned_to ?? 0) > 0)
		{
			$recipients = array_filter($recipients, function (User $user) use ($ticket) {
				return $user->id == $ticket->assigned_to;
			});
		}

		// Do not email the user who created the post
		$recipients = array_filter($recipients, function(User $user) use ($post) {
			return $user->id != $post->created_by;
		});

		// Get the ticket category parameters
		$category   = (new Category([]))->get($ticket->catid);
		$catParams  = new Registry(is_object($category) ? $category->params : '{}');

		// Get the manager IDs already added
		$managerIDs = array_map(fn(User $user) => (int) $user->id, $recipients);

		// Do I need to add “always notify” managers? This overrides all other options
		$recipients = $this->applyNotifyManagers($ticket, $recipients);

		// Do I need to remove managers? Removing managers overrides everything else, including explicitly added managers.
		$recipients = $this->applyExcludeManagers($ticket, $recipients);

		// Deduplicate the recipients
		$recipients = $this->deduplicateRecipients($recipients);

		// Send manager emails
		foreach ($recipients as $user)
		{
			$thisMailer = clone $mailer;

			$this->applyAddonEmails($user, $thisMailer);

			TemplateEmails::sendMail($emailKey, array_merge($emailData, [
				'user_name'     => $user->name,
				'user_email'    => $user->email,
				'user_username' => $user->username,
				'owner_groups'  => implode(', ', Permissions::getGroupsByUser($ticket->created_by)),
			]), $user, null, false, $thisMailer, false);
		}

		/**
		 * Notify the ticket owner in either of the following circumstances:
		 * - It is a new ticket
		 * - The post was created by a user other than the ticket owner
		 */
		if ($isNewTicket || !$createdByOwner)
		{
            $emailKey   =
                $newPost
                    ? sprintf("com_ats.owner_%s_%s", $ticketVisibilityForEmailKey, $suffixForEmailKey)
                    : 'com_ats.owner_edit';
            $user       = Permissions::getUser($ticket->created_by);
            $thisMailer = clone $mailer;

            $this->applyAddonEmails($user, $thisMailer);

            TemplateEmails::sendMail($emailKey, array_merge($emailData, [
                'user_name'     => $user->name,
                'user_email'    => $user->email,
                'user_username' => $user->username,
            ]), $user, null, false, $thisMailer);
		}

        // Finally, notify invited users (if any)
        $user_recipients = [];

        $emailKey   =
            $newPost
                ? sprintf("com_ats.owner_%s_%s", $ticketVisibilityForEmailKey, $suffixForEmailKey)
                : 'com_ats.owner_edit';

        $invited_users   = Permissions::getInvited($ticket->id);

        foreach ($invited_users as $invited_user)
        {
            $user_recipients[] = Permissions::getUser($invited_user);
        }

        // Do not email the user who created the post (it could be an invited user)
        $user_recipients = array_filter($user_recipients, function(User $user) use ($post) {
            return $user->id != $post->created_by;
        });

        foreach ($user_recipients as $user)
        {
            $thisMailer = clone $mailer;

            $this->applyAddonEmails($user, $thisMailer);

            TemplateEmails::sendMail($emailKey, array_merge($emailData, [
                'user_name'     => $user->name,
                'user_email'    => $user->email,
                'user_username' => $user->username,
            ]), $user, null, false, $thisMailer);
        }
	}

	private function applyNotifyManagers(TicketTable $ticket, array $recipients): array
	{
		// Get the ticket category parameters
		$category   = (new Category([]))->get($ticket->catid);
		$catParams  = new Registry(is_object($category) ? $category->params : '{}');

		// Get the manager IDs already added
		$managerIDs = array_map(fn(User $user) => (int) $user->id, $recipients);

		// Do I need to add “always notify” managers? This overrides all other options
		$notifyManagers = $catParams->get('notify_managers', []);
		$notifyManagers = is_array($notifyManagers) ? $notifyManagers : [];

		if (empty($notifyManagers) || in_array('all', $notifyManagers, true))
		{
			return $recipients;
		}

		$notifyManagers = array_map(fn($x) => (int) $x, $notifyManagers);
		$managerIDs     = array_intersect($notifyManagers, $managerIDs);
		$recipients     = [];

		if (empty($managerIDs))
		{
			return $recipients;
		}

		foreach ($managerIDs as $id)
		{
			if (!is_int($id))
			{
				continue;
			}

			$addUser = Permissions::getUser($id);

			if (($addUser === null) || ($addUser->id != $id))
			{
				continue;
			}

			$recipients[] = $addUser;
		}

		return $recipients;
	}

	private function deduplicateRecipients(array $recipients): array
	{
		$recipientIDs = [];

		return array_filter(
			$recipients,
			function (User $user) use (&$recipientIDs)
			{
				if (in_array($user->id, $recipientIDs))
				{
					return false;
				}

				$recipientIDs[] = $user->id;

				return true;
			}
		);
	}

	private function applyExcludeManagers(TicketTable $ticket, array $recipients): array
	{
		// Get the ticket category parameters
		$category   = (new Category([]))->get($ticket->catid);
		$catParams  = new Registry(is_object($category) ? $category->params : '{}');

		// Get the manager IDs already added
		$managerIDs = array_map(fn(User $user) => (int) $user->id, $recipients);

		// Do I need to remove managers? Removing managers overrides everything else, including explicitly added managers.
		$excludeManagers = $catParams->get('exclude_managers', []);
		$excludeManagers = is_array($excludeManagers) ? $excludeManagers : [];
		$excludeManagers = array_intersect($excludeManagers, $managerIDs);

		if (empty($excludeManagers))
		{
			return $recipients;
		}

		return array_filter($recipients, function (User $user) use ($excludeManagers) {
			return !in_array($user->id, $excludeManagers);
		});
	}

	/**
	 * Apply any add–on emails for the user to the mailer.
	 *
	 * Adds the add–on email addresses as CC
	 *
	 * @param   User  $user        The recipient user
	 * @param   Mail  $thisMailer  The mailer object to modify
	 *
	 * @return void
	 * @since  5.0.0
	 */
	private function applyAddonEmails($user, Mail $thisMailer): void
	{
		if (!class_exists(AddonEmails::class))
		{
			return;
		}

		$addOnEmails = AddonEmails::getAddonEmails($user->id);
		$addOnEmails = AddonEmails::filterAddonEmails($addOnEmails, $user->id);

		if (is_array($addOnEmails) && !empty($addOnEmails))
		{
			foreach ($addOnEmails as $email => $name)
			{
				try
				{
					$thisMailer->addCc($email, $name);
				}
				catch (Exception $e)
				{
				}
			}
		}
	}

	/**
	 * Attach files smaller than 2MB and create links to the rest of them
	 *
	 * @param   PostTable  $post       The post we are attaching files from
	 * @param   Mail       $mailer     The mailer which will be used to send the email
	 * @param   array      $emailData  The email template data (will be modified)
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	private function attachFiles(PostTable $post, Mail $mailer, array &$emailData): void
	{
		if (!defined('ATS_PRO') || !ATS_PRO)
		{
			return;
		}

		$attachmentURLs   = [];
		$attachments      = $post->getAttachments();
		$attachmentTokens = [];
		$numAttachments   = count(explode(',', $post->attachment_id ?? ''));
		$isOnlyAttachment = $numAttachments == 1;

		foreach ($attachments as $attachment)
		{
			$attFilename = $attachment->getAbsoluteFilename($attachment->mangled_filename);
			$realName    = $attachment->original_filename;

			if (!@file_exists($attFilename))
			{
				continue;
			}

			$parts       = explode('.', $realName);
			$extension   = strtolower(count($parts) > 1 ? $parts[count($parts) - 1] : '');
			$attSize     = @filesize($attFilename);
			$smallEnough = $attSize < 2097152;
			$isImageExt  = in_array($extension, ['png', 'gif', 'bmp', 'jpg', 'jpeg', 'webp', 'tif']);

			/**
			 * Inline a single attachment if it's under 2Mb and appears to be an image file
			 */
			if ($smallEnough && $isOnlyAttachment && $isImageExt)
			{
				try
				{
					$mailer->addAttachment($attFilename, $realName, 'base64', $attachment->mime_type);
				}
				catch (Exception $e)
				{
					$attachmentURLs[] = $realName;
				}
			}
			else
			{
				// Create a link to attachments not being inlined.
				$attachmentURLs[] = $realName;
			}
		}

		foreach ($attachmentURLs as $realName)
		{
			$attachmentTokens[] = "<li>" . Text::sprintf('COM_ATS_POST_LBL_ATTACHMENT_LINK', $realName) . "</li>";
		}

		if (!empty($attachmentURLs))
		{
			$emailData['attachment'] = "<ul>\n" . implode("\n", $attachmentTokens) . "</ul>";
		}
	}

	/**
	 * Get the category–specific email, as set in the category's options
	 *
	 * @param   int  $catId  The ID of the ATS category to get the email for
	 *
	 * @return  string|null  Category email, NULL if it's not applicable
	 *
	 * @since   5.0.0
	 */
	private function getCategoryEmail(int $catId): ?string
	{
		$db    = $this->getDatabase();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select($db->quoteName('params'))
			->from($db->quoteName('#__categories'))
			->where($db->quoteName('id') . ' = :id')
			->where($db->quoteName('extension') . ' = ' . $db->quote('com_ats'))
			->bind(':id', $catId, ParameterType::INTEGER);

		try
		{
			$paramsJson = $db->setQuery($query)->loadResult() ?: null;
		}
		catch (Exception $e)
		{
			return null;
		}

		if (empty($paramsJson))
		{
			return null;
		}

		$params = new Registry();
		$params->loadString($paramsJson, 'JSON');

		return trim($params->get('category_email', '')) ?: null;
	}

	/**
	 * Returns the very first post of a ticket
	 *
	 * @param   TicketTable  $ticket
	 *
	 * @return  PostTable|null  Very first post; null if it does not exist.
	 *
	 * @since   5.0.0
	 */
	private function getFirstPost(TicketTable $ticket): ?PostTable
	{
		$allPosts = $ticket->posts($ticket->id);

		if (empty($allPosts))
		{
			return null;
		}

		$firstPost = array_shift($allPosts);

		return (is_object($firstPost) && ($firstPost instanceof PostTable)) ? $firstPost : null;
	}

	/**
	 * Get a preconfigured mailer
	 *
	 * This methods:
	 * - Applies the correct priority to the mailer based on the ticket priority
	 * - Uses the correct Reply To address based on the category's email
	 *
	 * @param   TicketTable  $ticket
	 *
	 * @return Mail|null
	 *
	 * @throws Exception
	 * @since  5.0.0
	 */
	private function getMailer(TicketTable $ticket): ?Mail
	{
		/** @noinspection PhpDeprecationInspection */
		$mailer         = clone (class_exists(MailerFactoryInterface::class) ? Factory::getContainer()->get(
			MailerFactoryInterface::class)->createMailer() : Factory::getMailer());
		$app            = Factory::getApplication();
		$category_email = $this->getCategoryEmail($ticket->catid);

		if (!empty($category_email))
		{
			try
			{
				$mailer->addReplyTo($category_email, $app->get('fromname'));
			}
			catch (Exception $e)
			{
			}
		}

		$usePriorities    = ComponentHelper::getParams('com_ats')->get('ticketPriorities', 0) == 1;
		$mailer->Priority = 3;
		$head_priority    = 'Normal';

		// Set mail priority using phpmailer Priority variable and setting custom headers.
		// X-Priority header is already set by phpmailer
		// http://www.php.net/manual/en/function.mail.php#91058
		try
		{
			if ($usePriorities && ($ticket->priority == 1))
			{
				$mailer->Priority = 2;
				$head_priority    = 'High';
			}
			elseif ($usePriorities && ($ticket->priority > 5))
			{
				$mailer->Priority = 5;
				$head_priority    = 'Low';
			}
		}
		catch (Exception $e)
		{
		}

		try
		{
			/**
			 * So, the X-MSMail-Priority is required by Microsoft Outlook since it seems oblivious to the standard
			 * Importance header. However, SpamAssassin thinks that the presence of this header without X-MimeOLE means
			 * that your mail is spam. As a result we have to ignore users of Microsoft Outlook to make SpamAssassin not
			 * treat our legit mail as spam. If we don't, eventually all of the mails sent from the domain are marked as
			 * spam.
			 *
			 * As a small consolation prize, newer versions of Microsoft Outlook seem to have figured our how to honor
			 * RFC4021 in that respect. So even though we ignore users of legacy versions of Microsoft Outlook (which
			 * is broken) to please SpamAssassin (which is also broken) we can at least rest assured that users of a
			 * modern version of Microsoft Outlook will see the correct priority. Oh, well...
			 */
			// $mailer->addCustomHeader('X-MSMail-Priority: ' . $head_priority);
			$mailer->addCustomHeader('Importance: ' . $head_priority);
		}
		catch (Exception $e)
		{
		}

		return $mailer;
	}

	/**
	 * Get email parameter information for manager notes.
	 *
	 * @param   ManagernoteTable  $note    The manager note which we're sending email notifications about
	 * @param   TicketTable       $ticket  The ticket the note belongs to.
	 *
	 * @return  array  Information to pass onto Joomla's email templates manager
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function getNoteEmailData(ManagernoteTable $note, TicketTable $ticket): array
	{
		$app    = Factory::getApplication();
		$poster = Permissions::getUser($note->created_by);

		return [
			'sitename'        => $app->get('sitename', 'A Joomla! site'),
			'siteurl'         => ($app->isClient('site') || $app->isClient('administrator'))
				? Uri::base() : ComponentHelper::getParams('com_ats')->get('siteurl'),
			'user_name'       => '',
			'id'              => $ticket->id,
			'title'           => $ticket->title,
			'poster_name'     => $poster->name,
			'poster_username' => $poster->username,
			'poster_email'    => $poster->email,
			'catname'         => $ticket->getCategoryName(),
			'text'            => $note->note_html,
			'url'             => Route::link('site', 'index.php?option=com_ats&view=ticket&id=' . $ticket->id . '&catid=' . $ticket->catid, false, Route::TLS_IGNORE, true) . '#n' . $note->id,
		];
	}

	/**
	 * Returns the basic email template data for a given post and ticket
	 *
	 * @param   PostTable    $post    The post
	 * @param   TicketTable  $ticket  The ticket the post belongs to
	 *
	 * @return  array
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function getPostEmailData(PostTable $post, TicketTable $ticket): array
	{
		$app = Factory::getApplication();

		if ($post->created_by == -1)
		{
			$poster           = new User();
			$poster->username = 'system';
			$poster->name     = Text::_('COM_ATS_CLI_SYSTEMUSERLABEL');
			$poster->email    = 'system@example.com';
		}
		else
		{
			$poster = Permissions::getUser($post->created_by);
		}

		return [
			// TODO Remove replyafter / replyafter_html in version 6
			'replyafter'      => '',
			'replyafter_html' => '',
			'sitename'        => $app->get('sitename', 'A Joomla! site'),
			'siteurl'         => ($app->isClient('site') || $app->isClient('administrator'))
				? Uri::base() : ComponentHelper::getParams('com_ats')->get('siteurl'),
			'user_name'       => '',
			'id'              => $ticket->id,
			'title'           => $ticket->title,
			'poster_name'     => $poster->name,
			'poster_username' => $poster->username,
			'poster_email'    => $poster->email,
			'catname'         => $ticket->getCategoryName(),
			'text'            => URLTransformation::convertURLsToAbsolute($post->content_html),
			'url'             => Route::link('site', 'index.php?option=com_ats&view=ticket&id=' . $ticket->id . '&catid=' . $ticket->catid, false, Route::TLS_IGNORE, true) . '#p' . $post->id,
			'attachment'      => '',
		];
	}

	/**
	 * Returns the basic email template data for a ticket. Only used when a ticket is being assigned.
	 *
	 * @param   TicketTable  $ticket  The ticket
	 *
	 * @return  array
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function getTicketEmailData(TicketTable $ticket): array
	{
		$app      = Factory::getApplication();
		$assigner = Permissions::getUser();
		$assignee = Permissions::getUser($ticket->assigned_to);
		$post     = $this->getFirstPost($ticket);
		$postId   = ($post instanceof PostTable) ? $post->id : null;
		$postSuffix = $postId === null ? '' : sprintf('#p%d', $postId);

		if ($ticket->created_by == -1)
		{
			$poster           = new User();
			$poster->username = 'system';
			$poster->name     = Text::_('COM_ATS_CLI_SYSTEMUSERLABEL');
			$poster->email    = 'system@example.com';
		}
		else
		{
			$poster = Permissions::getUser($ticket->created_by);
		}

		return [
			// TODO Remove replyafter / replyafter_html in version 6
			'replyafter'        => '',
			'replyafter_html'   => '',
			'sitename'          => $app->get('sitename', 'A Joomla! site'),
			'siteurl'           => ($app->isClient('site') || $app->isClient('administrator'))
				? Uri::base() : ComponentHelper::getParams('com_ats')->get('siteurl'),
			'user_name'         => $assignee->name,
			'id'                => $ticket->id,
			'title'             => $ticket->title,
			'assigner_name'     => $assigner->name,
			'assigner_username' => $assigner->username,
			'assigner_email'    => $assigner->email,
			'poster_name'       => $poster->name,
			'poster_username'   => $poster->username,
			'poster_email'      => $poster->email,
			'catname'           => $ticket->getCategoryName(),
			'text'              => URLTransformation::convertURLsToAbsolute($post->content_html ?? ''),
			'url'               => Route::link(
					'site',
					sprintf('index.php?option=com_ats&view=ticket&id=%s&catid=%d', $ticket->id, $ticket->catid),
					false,
					Route::TLS_IGNORE,
					true
				) . $postSuffix,
			'attachment'        => '',
		];
	}

}