<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Model;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\Permissions;
use Akeeba\Component\ATS\Administrator\Mixin\ModelGetItemTrait;
use Akeeba\Component\ATS\Administrator\Table\PostTable;
use Akeeba\Component\ATS\Administrator\Table\TicketTable;
use Exception;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Form\FormFactoryInterface;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\MVC\Model\AdminModel;
use Joomla\CMS\Object\CMSObject;

#[\AllowDynamicProperties]
class PostModel extends AdminModel
{
	use ModelGetItemTrait;

	/**
	 * Flag set by PostController to indicate we are saving a guest ticket.
	 *
	 * This will use the privileges of the guest user instead of the created_by user of the ticket when processing the
	 * form.
	 *
	 * @var   bool
	 * @since 5.0.0
	 */
	private $guestTicket = false;

	public function __construct(
		$config = [], ?MVCFactoryInterface $factory = null, FormFactoryInterface $formFactory = null
	)
	{
		$this->guestTicket = (bool) ($config['guestTicket'] ?? false);

		parent::__construct($config, $factory, $formFactory);
	}


	/**
	 * @inheritDoc
	 */
	public function getForm($data = [], $loadData = true)
	{
		$id       = $data['id'] ?? null;
		$formName = empty($id) ? 'post_new' : 'post';

		$controlName = ($data['_control_name'] ?? null) ?: 'jform';

		if (isset($data['_control_name']))
		{
			unset($data['_control_name']);
		}

		$form = $this->loadForm(
			'com_ats.' . $formName,
			$formName,
			[
				'control'   => $controlName,
				'load_data' => $loadData,
			]
		) ?: false;

		if (empty($form))
		{
			return false;
		}

		// Get the Time Spent options
		$cParams            = ComponentHelper::getParams('com_ats');
		$timeSpentHidden    = $cParams->get('timespent_hide', 0) == 1;
		$timeSpentMandatory = !$timeSpentHidden && $cParams->get('timespent_mandatory', 0) != 1;

		// Modify the form based on access controls.
		$ticketId = $data['ticket_id'] ?? $form->getValue('ticket_id');
		/** @var TicketTable $ticket */
		$ticket = $this->getTable('Ticket', 'Administrator');
		$ticket->load($ticketId);

		if ($this->guestTicket)
		{
			$ticket->created_by = Permissions::getUser()->id;
		}

		$canDo = Permissions::getTicketPrivileges($ticket);
		$isPro = defined('ATS_PRO') && ATS_PRO;

		$fieldChanges = [
			'enabled'      => $canDo['edit.state'] ? '' : 'disable',
			'content_html' => $canDo['post'] ? '' : 'remove',
			'timespent'    => !$timeSpentHidden && $canDo['admin'] ? '' : 'remove',
			'origin'       => $canDo['admin'] ? '' : 'remove',
			'email_uid'    => $canDo['admin'] ? '' : 'remove',
			'created'      => $canDo['admin'] ? '' : 'remove',
			'created_by'   => $canDo['admin'] ? '' : 'remove',
			'modified'     => $canDo['admin'] ? '' : 'remove',
			'modified_by'  => $canDo['admin'] ? '' : 'remove',
			'attachments'  => ($canDo['attachment'] && $isPro) ? '' : 'remove',
			'usertags'     => ($canDo['admin'] && $isPro) ? '' : 'remove',
		];

		array_walk(
			$fieldChanges, function (string $type, string $fieldName, Form $form) {
			switch ($type)
			{
				case 'disable':
					$form->setFieldAttribute($fieldName, 'disabled', 'true');
					$form->setFieldAttribute($fieldName, 'required', 'false');
					$form->setFieldAttribute($fieldName, 'filter', 'unset');
					break;

				case 'remove':
					$form->removeField($fieldName);
					break;
			}
		}, $form
		);

		// Is timespent a required field?
		if (!$timeSpentMandatory)
		{
			$form->setFieldAttribute('timespent', 'required', 'false');
		}

		/**
		 * self::loadFormData() does not take $data into account. However, we use that to communicate overrides to the
		 * reply form, including the ticket ID. We need to make sure the $data array is taken into account indeed.
		 */
		if ($loadData)
		{
			$form->bind(array_merge($form->getData()->toArray(), $data));
		}

		return $form;
	}

	/**
	 * Method to get a single record.
	 *
	 * @param   int|null  $pk  The id of the primary key.
	 *
	 * @return  PostTable|bool  Object on success, false on failure.
	 * @throws  Exception
	 * @since        5.0.0
	 *
	 * @noinspection PhpMissingParentCallCommonInspection
	 */
	public function getItem($pk = null)
	{
		return $this->getItemTable($pk);
	}

	public function validate($form, $data, $group = null)
	{
		// Set the state variable marking whether this is a new record
		$this->setState('post.isNew', empty($data['id'] ?? null));

		$isValid = parent::validate($form, $data, $group);

		if (!$isValid)
		{
			return $isValid;
		}

		// Post size check — TODO Make this a form Rule
		$isManager = Permissions::isManager($data['catid'] ?? null);

		if (!$isManager)
		{
			$maxSize  = ComponentHelper::getParams('com_ats')->get('maxBodySize', 128) ?: 128;
			$maxSize  = is_numeric($maxSize) ? intval($maxSize) : 128;
			$maxSize  = 1024 * max(min($maxSize, 10240), 1);
			$postSize = strlen($data['content_html']);

			if ($postSize > $maxSize)
			{
				if (!method_exists($this, 'setError'))
				{
					throw new \RuntimeException(Text::_('COM_ATS_POST_ERR_BODY_TOO_BIG'));
				}

				/** @noinspection PhpDeprecationInspection */
				$this->setError(Text::_('COM_ATS_POST_ERR_BODY_TOO_BIG'));

				$isValid = false;
			}
		}

		return $isValid;
	}

	/** @inheritdoc */
	protected function canDelete($record)
	{
		/** @var TicketTable $ticket */
		$ticket = $this->getTable('Ticket', 'Administrator');

		if (!empty($record) && isset($record->ticket_id) && empty($record->ticket_id))
		{
			$ticket->load($record->ticket_id);
			$record->setTicket($ticket);
		}

		$canDo = Permissions::getPostPrivileges($record);

		return $canDo['delete'];
	}

	/** @inheritdoc */
	protected function canEditState($record)
	{
		/** @var TicketTable $ticket */
		$ticket = $this->getTable('Ticket', 'Administrator');

		if (!empty($record) && isset($record->ticket_id) && empty($record->ticket_id))
		{
			$ticket->load($record->ticket_id);
			$record->setTicket($ticket);
		}

		$canDo = Permissions::getPostPrivileges($record);

		return $canDo['edit.state'];
	}

	/**
	 * Load the data of an add / edit form.
	 *
	 * The data is loaded from the user state. If the user state is empty we load the item being edited. If there is no
	 * item being edited we will override the default table values with the respective list filter values. This makes
	 * sense for users. If I am filtering by category X and maturity Stable I am probably trying to see if there is a
	 * specific stable version released in category X and, if not, create it. Using the filter values reduces the
	 * possibility for silly mistakes on the part of the operator.
	 *
	 * @return array|bool|CMSObject|mixed
	 * @throws Exception
	 */
	protected function loadFormData()
	{
		/** @var CMSApplication $app */
		$app  = Factory::getApplication();
		$data = $app->getUserState('com_ats.edit.post.data', []);

		if (empty($data))
		{
			$data = (object) $this->getItem()->getProperties();
		}

		$this->preprocessData('com_ats.post', $data);

		return $data;
	}

	/** @inheritdoc */
	protected function prepareTable($table)
	{
		// Set up the created / modified date
		$date  = Factory::getDate();
		$user  = Factory::getApplication()->getIdentity();
		$isNew = empty($table->getId());

		if ($isNew)
		{
			// Set the values
			$table->created    = $date->toSql();
			$table->created_by = $user->id;
		}
		else
		{
			// Set the values
			$table->modified    = $date->toSql();
			$table->modified_by = $user->id;
		}
	}

}