<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Model;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\Permissions;
use Akeeba\Component\ATS\Administrator\Mixin\CMSObjectWorkaroundTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ModelGetItemTrait;
use Akeeba\Component\ATS\Administrator\Mixin\RunPluginsTrait;
use Akeeba\Component\ATS\Administrator\Table\TicketTable;
use Akeeba\Component\ATS\Site\Service\Category;
use Exception;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Event\Model\BeforeBatchEvent;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Helper\TagsHelper;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Menu\AbstractMenu;
use Joomla\CMS\Menu\MenuItem;
use Joomla\CMS\MVC\Model\AdminModel;
use Joomla\CMS\Object\CMSObject;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Table\User;
use Joomla\CMS\User\UserFactoryInterface;
use Joomla\CMS\User\UserHelper;
use Joomla\Component\Fields\Administrator\Helper\FieldsHelper;
use Joomla\Component\Fields\Administrator\Model\FieldModel;
use Joomla\Component\Users\Site\Model\RegistrationModel;
use Joomla\Database\DatabaseDriver;
use Joomla\Database\ParameterType;
use Joomla\Registry\Registry;
use ReflectionObject;
use stdClass;
use function count;

#[\AllowDynamicProperties]
class TicketModel extends AdminModel
{
	use ModelGetItemTrait;
	use RunPluginsTrait;
	use CMSObjectWorkaroundTrait;

	/**
	 * Allowed batch commands
	 *
	 * @var  array
	 */
	protected $batch_commands = [
		'tag'         => 'batchTag',
		'category_id' => 'batchMove',
		'assigned_to' => 'batchAssignTo',
		'status'      => 'batchStatus',
		'priority'    => 'batchPriority',
	];

	/**
	 * Batch copy/move command. If set to false, the batch copy/move command is not supported.
	 *
	 * We set it to false because we map batch[category_id] to the batchMove command instead of letting AdminModel
	 * decide whether to use batchMove or batchCopy based on batch[move_copy].
	 *
	 * @var    string
	 * @since  7.0
	 */
	protected $batch_copymove = false;

	/**
	 * Method to change the status of one or more records.
	 *
	 * @param   array    &$pks    A list of the primary keys to change.
	 * @param   string    $value  The new ticket status
	 *
	 * @return  bool  True on success.
	 *
	 * @since   5.0.0
	 */
	public function closeOpenTicket(&$pks, $value = 'C')
	{
		$user = Factory::getApplication()->getIdentity();
		/** @var TicketTable $table */
		$table = $this->getTable();
		$pks   = (array) $pks;

		if (!in_array($value, ['C', 'O']))
		{
			Log::add(Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'), Log::WARNING, 'jerror');

			return false;
		}

		// Access checks.
		foreach ($pks as $i => $pk)
		{
			$table->reset();

			if ($table->load($pk))
			{
				$canDo      = Permissions::getTicketPrivileges($table, $user);
				$permission = ($value === 'C') ? 'close' : 'admin';

				if (!$canDo[$permission])
				{
					// Prune items that you can't change.
					unset($pks[$i]);

					Log::add(Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'), Log::WARNING, 'jerror');

					return false;
				}

				/**
				 * Prune items that are already at the given state.  Note: Only models whose table correctly
				 * sets 'published' column alias (if different than published) will benefit from this
				 */
				$statusColumnName = $table->getColumnAlias('status');

				if (property_exists($table, $statusColumnName) && ($table->{$statusColumnName} ?? $value) == $value)
				{
					unset($pks[$i]);

					continue;
				}
			}
		}

		// Check if there are items to change
		if (!count($pks))
		{
			return true;
		}

		// Attempt to change the state of the records.
		[$didChangeStatus, $error,] = $this->cmsObjectSafeCall($table, 'changeStatus', $pks, $value, $user->id);

		if (!$didChangeStatus)
		{
			$this->setErrorOrThrow($error);

			return false;
		}

		// Clear the component's cache
		$this->cleanCache();

		return true;
	}

	/**
	 * Create a new Joomla user
	 *
	 * @param   string|null  $name      Full name of the user
	 * @param   string|null  $username  Desired username
	 * @param   string|null  $email     Desired email address
	 *
	 * @return  false|int User ID, or FALSE on failure
	 *
	 * @throws Exception
	 * @since  5.0.0
	 */
	public function createUser(?string $name, ?string $username = null, ?string $email = null)
	{
		// Change the behavior of com_users for the purpose of creating a new user through ATS.
		$cParams = ComponentHelper::getParams('com_users');

		$json = $cParams->toString('JSON');

		// -- Allow new users to be created
		$cParams->set('allowUserRegistration', 1);
		// -- Send users their auto–generated password
		$cParams->set('sendpassword', 1);
		// -- No CAPTCHA necessary
		$cParams->set('captcha', '0');
		// -- Fake password requirements so that our random password always works
		$cParams->set('minimum_length', '8');
		$cParams->set('minimum_integers', '0');
		$cParams->set('minimum_symbols', '0');
		$cParams->set('minimum_uppercase', '0');
		$cParams->set('minimum_lowercase', '0');

		// -- Do I have to force self–activation of new users?
		if (ComponentHelper::getParams('com_ats')->get('forceGuestActivation', 0) == 1)
		{
			$cParams->set('useractivation', 1);
		}

		$app  = Factory::getApplication();
		$data = [
			'name'      => $name ?: $username,
			'username'  => $username,
			'password1' => UserHelper::genRandomPassword(24),
			'email1'    => $email,
			'groups'    => [$cParams->get('new_usertype', 2)],
			'params'    => [],
		];

		$data['email2']    = $data['email2'] ?? $data['email1'];
		$data['password2'] = $data['password2'] ?? $data['password1'];

		$comUsersPath = JPATH_SITE . '/components/com_users';

		Form::addFormPath($comUsersPath . '/forms');
		Form::addFormPath($comUsersPath . '/models/forms');
		Form::addFieldPath($comUsersPath . '/models/fields');
		Form::addFormPath($comUsersPath . '/model/form');
		Form::addFieldPath($comUsersPath . '/model/field');

		$app->getLanguage()->load('com_users');

		/** @var RegistrationModel $registrationModel */
		$registrationModel = $app->bootComponent('com_users')->getMVCFactory()
			->createModel('Registration', 'Site', ['ignore_request' => true]);

		// We don't want the registration model to go through the request. So, we have to TRICK IT!
		$userParams     = ComponentHelper::getParams('com_users');
		$fakeDataObject = (object) [
			'name'     => $data['name'],
			'username' => $data['username'],
			'email1'   => $data['email1'],
			'groups'   => [
				$userParams->get('new_usertype', $userParams->get('guest_usergroup', 1)),
			],
		];
		$refObject      = new ReflectionObject($registrationModel);
		$refProp        = $refObject->getProperty('data');

		if (version_compare(PHP_VERSION, '8.1.0', 'lt'))
		{
			$refProp->setAccessible(true);
		}

		$refProp->setValue($registrationModel, $fakeDataObject);

		[$return, $error,] = $this->cmsObjectSafeCall($registrationModel, 'register', $data);

		$cParams->loadString($json);

		/**
		 * Joomla's registration model can return on of the following:
		 * - false on error. We set the error in the model and return it.
		 * - the User ID if a user was created and immediately activated. We return it as is.
		 * - the string 'useractivate' or 'adminactivate' is self or administrator account activation is required.
		 *
		 * The latter string return values are useless. So we have to look for the user account by username, load it and
		 * return its ID.
		 */

		// False means error. Return as–is.
		if ($return === false)
		{
			$this->setErrorOrThrow($error);

			return false;
		}

		// Already activated user created. Return as–is.
		if (is_integer($return))
		{
			return $return;
		}

		// A user that needs registration was created. Find the user ID and return it.
		$user = Factory::getContainer()->get(UserFactoryInterface::class)->loadUserByUsername($username);

		return $user->id;
	}

	public function inviteUser(int $ticket_id, string $username)
	{
		if (!$ticket_id || !$username)
		{
			throw new \RuntimeException(Text::_('COM_ATS_TICKET_INVITE_MISSING_DATA'));
		}

		/** @var TicketTable $ticket_table */
		$ticket_table = $this->getTable();
		$ticket_table->load($ticket_id);

		$user = Factory::getContainer()->get(UserFactoryInterface::class)->loadUserByUsername($username);

		if (!$ticket_table->getId() || !$user->id)
		{
			throw new \RuntimeException(Text::_('COM_ATS_TICKET_INVITE_INVALID_DATA'));
		}

		// Peform some sanity checks before inviting the user
		$cParams      = ComponentHelper::getParams('com_ats');
		$limitInvites = $cParams->get('invite_limit', 10);

		$db    = $this->getDatabase();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('*')
			->from($db->qn('#__ats_tickets_users'))
			->where($db->qn('ticket_id') . ' = ' . $db->q($ticket_id));
		$users = $db->setQuery($query)->loadObjectList('user_id');

		if (isset($users[$user->id]))
		{
			throw new \RuntimeException(Text::_('COM_ATS_TICKET_INVITE_ALREADY_INVITED'));
		}

		if (count($users) >= $limitInvites)
		{
			throw new \RuntimeException(Text::_('COM_ATS_TICKET_INVITE_TOO_MANY'));
		}

		// Do not allow inviting users that already have read/write permissions
		$permissions = Permissions::getTicketPrivileges($ticket_table, $user);

		if ($permissions['view'] && $permissions['post'])
		{
			throw new \RuntimeException(Text::_('COM_ATS_TICKET_INVITE_ALREADY_HAVE_ACCESS'));
		}

		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->insert($db->qn('#__ats_tickets_users'))
			->columns(['ticket_id', 'user_id'])
			->values($db->q($ticket_id) . ', ' . $db->q($user->id));
		$db->setQuery($query)->execute();
	}

	public function removeInvite(int $invited_user, int $ticket_id)
	{
		if (!$invited_user || !$ticket_id)
		{
			throw new \RuntimeException(Text::_('COM_ATS_TICKET_INVITE_MISSING_DATA'));
		}

		$db    = $this->getDatabase();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->delete($db->qn('#__ats_tickets_users'))
			->where($db->qn('ticket_id') . ' = ' . $db->q($ticket_id))
			->where($db->qn('user_id') . ' = ' . $db->q($invited_user));
		$db->setQuery($query)->execute();
	}

	/**
	 * @inheritDoc
	 */
	public function getForm($data = [], $loadData = true)
	{
		$id        = $data['id'] ?? null;
		$newTicket = empty($id);
		$formName  = $newTicket ? 'ticket_new' : 'ticket';

		$controlName = ($data['_control_name'] ?? null) ?: 'jform';

		if (isset($data['_control_name']))
		{
			unset($data['_control_name']);
		}

		if ($newTicket && !empty($data) && is_null($data['typeAlias'] ?? null))
		{
			/** @var CMSApplication $app */
			$app = Factory::getApplication();

			$app->setUserState('com_ats.edit.ticket.data', $data);
		}

		/**
		 * When saving a ticket as a regular user I need to load the data in the form regardless of the $loadData
		 * parameter. This is the only way to ensure that a hidden trigger field, used to control the display of other
		 * fields through onload, will have a non-default value. Otherwise, that trigger field might have a default
		 * value which causes the controlled fields to not display. If a field is set to not display it will NOT be
		 * included in the form definition by the plg_system_fields plugin. This means that the FromModel's validate()
		 * method will prune the values submitted by the user for this field, therefore the field will never be saved.
		 *
		 * This is exactly as convoluted as it sounds.
		 */
		$appInput = Factory::getApplication()->getInput();

		if ($appInput->getInt('id') > 0 && $appInput->getCmd('task') === 'save')
		{
			$loadData = true;
		}

		/**
		 * IMPORTANT! The name of the form MUST be 'com_ats.ticket' even when we load the new ticket form (ticket_new).
		 * The name of the form is used as the context which, in turn, is used to determine the custom fields to
		 * display in the HTML form. Since the custom fields are keyed to the context 'com_ats.ticket' we need to use
		 * that form name at all times. It makes perfect sense when you think about it!
		 */
		$form = $this->loadForm(
			'com_ats.ticket',
			$formName,
			[
				'control'   => $controlName,
				'load_data' => $loadData,
			]
		) ?: false;

		if (empty($form))
		{
			return false;
		}

		$id = $data['id'] ?? $form->getValue('id');

		$cParams       = ComponentHelper::getParams('com_ats');
		$hasPriorities = $cParams->get('ticketPriorities', 0) == 1;
		$isFrontend    = Factory::getApplication()->isClient('site');
		$item          = $this->getItem($id);
		$canEditState  = empty($item) ? false : $this->canEditState((object) $item);
		$catId         = ($data['catid'] ?? $form->getValue('catid')) ?: null;
		$isManager     = Permissions::isManager($catId);

		// Remove `priority` if the feature is disabled
		if (!$hasPriorities)
		{
			$form->removeField('priority');
		}

		// Handle the public field depending on user privileges and category options.
		$acls         = Permissions::getAclPrivileges($catId);
		$catParams    = $this->getCatParams($catId);
		$forceType    = $catParams->get('forcetype', '');
		$canDoPrivate = $acls['ats.private'] || $isManager;
		$hidePublic   = $cParams->get('hide_public', 0) == 1;

		if (!$canDoPrivate)
		{
			// The user can't file private tickets; remove the field
			$form->removeField('public');
		}
		elseif ($forceType === 'PRIV' && $hidePublic)
		{
			$form->setFieldAttribute('public', 'disabled', 'true');
			$form->setFieldAttribute('public', 'required', 'false');
			$form->setFieldAttribute('public', 'filter', 'unset');
			$form->setFieldAttribute('public', 'type', 'hidden');

			if ($loadData)
			{
				$form->setValue('public', null, 0);
			}
		}
		elseif (!empty($forceType))
		{
			// The user is forced to file a ticket with a specific visibility
			$form->setFieldAttribute('public', 'disabled', 'true');
			$form->setFieldAttribute('public', 'required', 'false');
			$form->setFieldAttribute('public', 'filter', 'unset');

			if ($loadData)
			{
				$form->setValue('public', null, ($forceType == 'PRIV') ? 0 : 1);
			}
		}

		// Modify the form based on access controls and whether we are in the frontend.
		if ($isFrontend && !empty($id) && !Permissions::isManager($catId))
		{
			$removeFields = [
				'id',
				'alias',
				'enabled',
				'assigned_to',
				'tags',
				'origin',
				'timespent',
				'created',
				'created_by',
				'modified',
				'modified_by',
			];

			$perms = Permissions::getAclPrivileges($catId);

			if (!$perms['core.edit.state'])
			{
				$removeFields[] = 'public';
				$removeFields[] = 'status';
			}

			foreach ($removeFields as $fieldName)
			{
				$form->removeField($fieldName);
			}

			foreach (['created'] as $fieldName)
			{
				$form->setFieldAttribute($fieldName, 'disabled', 'true');
				$form->setFieldAttribute($fieldName, 'required', 'false');
				$form->setFieldAttribute($fieldName, 'filter', 'unset');
			}

			foreach (['catid'] as $fieldName)
			{
				$form->setFieldAttribute($fieldName, 'label', '');
				$form->setFieldAttribute($fieldName, 'class', 'd-none');
				$form->setFieldAttribute($fieldName, 'required', false);
				$form->setFieldAttribute($fieldName, 'readonly', 'true');
			}
		}
		elseif (!$newTicket && !$canEditState)
		{
			foreach (
				[
					'enabled',
					'status',
					'public',
					'priority',
					'assigned_to',
					'catid',
					'tags',
					'origin',
					'timespent',
					'created',
				] as $fieldName
			)
			{
				$form->setFieldAttribute($fieldName, 'disabled', 'true');
				$form->setFieldAttribute($fieldName, 'required', 'false');
				$form->setFieldAttribute($fieldName, 'filter', 'unset');
			}
		}
		elseif (!$newTicket)
		{
			$timeSpentHidden = $cParams->get('timespent_hide', 0) == 1;

			if ($timeSpentHidden)
			{
				$form->removeField('timespent');
			}
		}

		/**
		 * When saving a new ticket I do NOT want the post–specific fields to interfere with my saving of the ticket
		 * data itself. Therefore I need to exclude them from the form.
		 */
		if ($this->getState('ticket.savingNew', false))
		{
			$fields = $form->getFieldset('post');

			foreach ($fields as $field)
			{
				$form->removeField($field->fieldname);
			}
		}

		// Special form processing for new tickets
		if ($newTicket)
		{
			$this->processNewTicketForm($form);
		}

		if ($loadData && is_array($data) && isset($data['params']) && !$newTicket)
		{
			$this->migrateLegacyCustomFieldsData($data['params'], $form, $item);
		}

		return $form;
	}

	/**
	 * Method to get a single record.
	 *
	 * @param   int|null  $pk  The id of the primary key.
	 *
	 * @return  TicketTable|bool  Object on success, false on failure.
	 * @throws  Exception
	 * @since        5.0.0
	 *
	 * @noinspection PhpMissingParentCallCommonInspection
	 */
	public function getItem($pk = null)
	{
		$item = $this->getItemTable($pk);

		// Load item tags
		if (!empty($item->id))
		{
			$item->tags = new TagsHelper();
			$item->tags->getTagIds($item->id, 'com_ats.ticket');
		}


		return $item;
	}

	public function getInvitedUsers($pk = null)
	{
		$item = $this->getItemTable($pk);

		// Load item tags
		if (empty($item->id))
		{
			return [];
		}

		$db    = $this->getDatabase();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('user_id')
			->from($db->qn('#__ats_tickets_users'))
			->where($db->qn('ticket_id') . ' = ' . $db->q($item->id));
		$users = $db->setQuery($query)->loadColumn();

		$jusers = [];

		$userFactory = Factory::getContainer()->get(\Joomla\CMS\User\UserFactoryInterface::class);

		foreach ($users as $user)
		{
			$jusers[] = $userFactory->loadUserById($user);
		}

		return $jusers;
	}

	/**
	 * Method to change the public state of one or more records.
	 *
	 * @param   array    &$pks    A list of the primary keys to change.
	 * @param   int       $value  The value of the public state.
	 *
	 * @return  bool  True on success.
	 *
	 * @since   5.0.0
	 */
	public function makepublic(&$pks, $value = 1)
	{
		$user = Factory::getApplication()->getIdentity();
		/** @var TicketTable $table */
		$table = $this->getTable();
		$pks   = (array) $pks;

		$context = $this->option . '.' . $this->name;

		// Include the plugins for the change of state event.
		PluginHelper::importPlugin($this->events_map['change_state']);

		// Access checks.
		foreach ($pks as $i => $pk)
		{
			$table->reset();

			if ($table->load($pk))
			{
				if (!$this->canEditState($table))
				{
					// Prune items that you can't change.
					unset($pks[$i]);

					Log::add(Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'), Log::WARNING, 'jerror');

					return false;
				}

				/**
				 * Prune items that are already at the given state.  Note: Only models whose table correctly
				 * sets 'published' column alias (if different than published) will benefit from this
				 */
				$publicColumnName = $table->getColumnAlias('public');

				if (property_exists($table, $publicColumnName) && ($table->{$publicColumnName} ?? $value) == $value)
				{
					unset($pks[$i]);

					continue;
				}
			}
		}

		// Check if there are items to change
		if (!count($pks))
		{
			return true;
		}

		// Attempt to change the state of the records.
		[$madePublic, $error, $errors] = $this->cmsObjectSafeCall($table, 'makepublic', $pks, $value, $user->id);

		if (!$madePublic)
		{
			$this->setErrorOrThrow($error);

			return false;
		}

		// Clear the component's cache
		$this->cleanCache();

		return true;
	}

	/** @inheritdoc */
	public function save($data)
	{
		// Support for guest tickets
		$user      = Permissions::getUser();
		$table     = $this->getTable();
		$key       = $table->getKeyName();
		$pk        = (isset($data[$key])) ? $data[$key] : (int) $this->getState($this->getName() . '.id');
		$isNew     = (int) $pk <= 0;
		$catId     = ($data['catid'] ?? $table->catid) ?: null;
		$isManager = Permissions::isManager($catId);
		$acls      = Permissions::getAclPrivileges($catId);
		$catParams = $this->getCatParams($catId);
		$forceType = $catParams->get('forcetype', '');

		$this->setState('guestTicket', false);

		if ($isNew && $user->guest)
		{
			$name     = $data['name'] ?? null;
			$username = $data['username'] ?? null;
			$email    = $data['email'] ?? null;

			unset($data['name']);
			unset($data['username']);
			unset($data['email']);

			$data['created_by'] = $this->createUser($name, $username, $email);

			if ($data['created_by'] === false)
			{
				return false;
			}

			/**
			 * Override user tells self::prepareTable() to not set a created_by field. This prevents prepareTable from
			 * setting the created_by field value to 0.
			 *
			 * No permissions check tells the TicketTable to not perform any checks before storing the submitted ticket
			 * information. This is necessary because the site owner may have allowed the Guest group to create new
			 * tickets BUT NOT the Registered group. Since the newly created user is Registered we'd end up unable to
			 * save the ticket.
			 */
			TicketTable::$overrideuser       = true;
			TicketTable::$noPermissionsCheck = true;

			// Tell the TicketController this is a guest ticket so it can tell the PostController to perform no checks.
			$this->setState('guestTicket', true);
		}
		elseif ($isNew && $isManager && !empty($data['created_by'] ?? null))
		{
			// We have a user override when creating the ticket.
			TicketTable::$overrideuser = true;
		}

		// Handle the public field depending on user privileges and category options.
		$canDoPrivate = $acls['ats.private'] || $isManager;

		if (!$canDoPrivate)
		{
			// The user can't file private tickets; remove the field
			$data['public'] = null;
		}
		elseif (!empty($forceType))
		{
			$data['public'] = ($forceType == 'PRIV') ? 0 : 1;
		}

		$return = parent::save($data);

		/**
		 * If saving the ticket has failed I need to delete the user I just created. Otherwise the guest will not be
		 * able to submit the ticket unless they log in and try all over again — not exactly a pleasant experience!
		 *
		 * I am going directly through the User table instead of the model since this is a brand new user. My reasonable
		 * expectation is that no third party extension has initialised any data for this user and if it has it can
		 * survive when the user record is actually deleted.
		 */
		if ($this->getState('guestTicket', false) && !$return)
		{
			/** @var DatabaseDriver $db */
			$db     = $this->getDatabase();
			$killMe = new User($db);
			try
			{
				$killMe->delete($data['created_by']);
			}
			catch (Exception $e)
			{
				// If a database error occurred I'm out of luck and out of ideas...
			}
		}

		TicketTable::$overrideuser       = false;
		TicketTable::$noPermissionsCheck = false;

		return $return;
	}

	/**
	 * Are we being called from a frontend page whose manu item is New Ticket with a **specific** category ID selected?
	 *
	 * @return  bool
	 *
	 * @since   5.0.3
	 */
	public function isFrontendNewTicketForCategory(): bool
	{
		// Condition: we must be in the frontend
		try
		{
			/** @var SiteApplication $app */
			$app = Factory::getApplication();
		}
		catch (Exception $e)
		{
			return false;
		}

		if (!$app->isClient('site'))
		{
			return false;
		}

		// Condition: must have active menu item
		$menu       = $app->getMenu();
		$activeItem = ($menu instanceof AbstractMenu) ? $menu->getActive() : null;

		if (!($activeItem instanceof MenuItem))
		{
			return false;
		}

		// Condition: the request MUST NOT be overriding the task
		$input = $app->getInput();
		$task  = $input->get('task', '');

		if (!empty($task) && strpos($task, 'display') === false && strpos($task, 'main') === false)
		{
			return false;
		}

		// Condition: must be a New Ticket menu item with a non-zero ID
		$query  = $activeItem->query;
		$option = $query['option'] ?? '';
		$view   = $query['view'] ?? '';
		$layout = $query['layout'] ?? '';
		$id     = $query['id'] ?? '';

		return $option === 'com_ats' && $view === 'ticket' && $layout === 'newticket' && is_numeric($id) && !empty($id);
	}

	/**
	 * Batch assign tickets to a manager
	 *
	 * @param   string|int  $value     The user ID to assign them to, <=0 to unassign, '' to do nothing
	 * @param   array       $pks       An array of ticket IDs.
	 * @param   array       $contexts  An array of item contexts.
	 *
	 * @return  bool  True if successful, false otherwise and internal error is set.
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 */
	protected function batchAssignTo($value, array $pks, array $contexts): bool
	{
		if (!is_numeric($value))
		{
			return true;
		}

		// Initialize re-usable member properties, and re-usable local variables
		$this->initBatch();

		foreach ($pks as $pk)
		{
			$this->table->reset();
			$this->table->load($pk);

			if (!Permissions::isManager($this->table->catid))
			{
				$this->setErrorOrThrow(Text::_('JLIB_APPLICATION_ERROR_BATCH_CANNOT_EDIT'));

				return false;
			}

			$this->table->assigned_to = (int) $value <= 0 ? null : (int) $value;

			$this->triggerPluginEvent(
				$this->event_before_batch, ['src' => $this->table, 'type' => 'ats_assign_to'], BeforeBatchEvent::class
			);

			// Check the row.
			[$isChecked, $error,] = $this->cmsObjectSafeCall($this->table, 'check');

			if (!$isChecked)
			{
				$this->setErrorOrThrow($error);

				return false;
			}

			[$isStored, $error,] = $this->cmsObjectSafeCall($this->table, 'store');
			if (!$isStored)
			{
				$this->setErrorOrThrow($error);

				return false;
			}
		}

		// Clean the cache
		$this->cleanCache();

		return true;
	}

	/**
	 * Batch change the priority of a ticket
	 *
	 * @param   string|int  $value     The priority of the ticket: 0 to 10, 0 being highest.
	 * @param   array       $pks       An array of ticket IDs.
	 * @param   array       $contexts  An array of item contexts.
	 *
	 * @return  bool  True if successful, false otherwise and internal error is set.
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 */
	protected function batchPriority($value, array $pks, array $contexts): bool
	{
		if (!is_numeric($value))
		{
			return true;
		}

		$value = max(0, min((int) $value, 10));

		// Initialize re-usable member properties, and re-usable local variables
		$this->initBatch();

		$user = Factory::getApplication()->getIdentity();

		foreach ($pks as $pk)
		{
			$this->table->reset();
			$this->table->load($pk);

			$canEditState = Permissions::isManager($this->table->catid)
			                || $user->authorise('core.edit.state', 'com_ats.category.' . $this->table->catid);

			if (!$canEditState)
			{
				$this->setErrorOrThrow(Text::_('JLIB_APPLICATION_ERROR_BATCH_CANNOT_EDIT'));

				return false;
			}

			$this->table->priority = $value;

			$this->triggerPluginEvent(
				$this->event_before_batch, ['src' => $this->table, 'type' => 'ats_priority'], BeforeBatchEvent::class
			);

			// Check the row.
			[$isChecked, $error,] = $this->cmsObjectSafeCall($this->table, 'check');
			if (!$isChecked)
			{
				$this->setErrorOrThrow($error);

				return false;
			}

			[$isStored, $error,] = $this->cmsObjectSafeCall($this->table, 'store');
			if (!$isStored)
			{
				$this->setErrorOrThrow($error);

				return false;
			}
		}

		// Clean the cache
		$this->cleanCache();

		return true;
	}

	/**
	 * Batch change the status of a ticket
	 *
	 * @param   string  $value     The status of the ticket. Must be O, P, C or one of the custom statuses.
	 * @param   array   $pks       An array of ticket IDs.
	 * @param   array   $contexts  An array of item contexts.
	 *
	 * @return  bool  True if successful, false otherwise and internal error is set.
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 */
	protected function batchStatus($value, array $pks, array $contexts): bool
	{
		if (empty($value))
		{
			return true;
		}

		$validStatuses = Permissions::getStatuses();
		$value         = strtoupper($value);

		if (!array_key_exists($value, $validStatuses))
		{
			$this->setErrorOrThrow(Text::_('COM_ATS_TICKETS_ERR_INVALIDSTATUS'));

			return false;
		}

		// Initialize re-usable member properties, and re-usable local variables
		$this->initBatch();

		$user = Factory::getApplication()->getIdentity();

		foreach ($pks as $pk)
		{
			$this->table->reset();
			$this->table->load($pk);

			$canEditState = Permissions::isManager($this->table->catid)
			                || $user->authorise('core.edit.state', 'com_ats.category.' . $this->table->catid);

			if (!$canEditState)
			{
				$this->setErrorOrThrow(Text::_('JLIB_APPLICATION_ERROR_BATCH_CANNOT_EDIT'));

				return false;
			}

			$this->table->status = $value;

			$this->triggerPluginEvent(
				$this->event_before_batch, ['src' => $this->table, 'type' => 'ats_status'], BeforeBatchEvent::class
			);


			// Check the row.
			[$isChecked, $error,] = $this->cmsObjectSafeCall($this->table, 'check');
			if (!$isChecked)
			{
				$this->setErrorOrThrow($error);

				return false;
			}

			[$isStored, $error,] = $this->cmsObjectSafeCall($this->table, 'store');
			if (!$isStored)
			{
				$this->setErrorOrThrow($error);

				return false;
			}
		}

		// Clean the cache
		$this->cleanCache();

		return true;
	}

	protected function canDelete($record)
	{
		$canDo = Permissions::getTicketPrivileges($record);

		return $canDo['delete'];
	}

	/**
	 * Method to test whether a record can have its state changed.
	 *
	 * @param   object  $record  A record object.
	 *
	 * @return  bool  True if allowed to change the state of the record. Defaults to the permission for the
	 *                   component.
	 *
	 * @since   5.0.0
	 */
	protected function canEditState($record)
	{
		$canDo = Permissions::getTicketPrivileges($record);

		return $canDo['edit.state'];
	}

	/**
	 * Load the data of an add / edit form.
	 *
	 * The data is loaded from the user state. If the user state is empty we load the item being edited. If there is no
	 * item being edited we will override the default table values with the respective list filter values. This makes
	 * sense for users. If I am filtering by category X and maturity Stable I am probably trying to see if there is a
	 * specific stable version released in category X and, if not, create it. Using the filter values reduces the
	 * possibility for silly mistakes on the part of the operator.
	 *
	 * @return array|bool|CMSObject|mixed
	 * @throws Exception
	 */
	protected function loadFormData()
	{
		/** @var CMSApplication $app */
		$app           = Factory::getApplication();
		$data          = $app->getUserState('com_ats.edit.ticket.data', []);
		$noSavedData   = empty($data);
		$data['catid'] = $this->getState('ticket.catid', null) ?? ($data['catid'] ?? null);

		if ($noSavedData)
		{
			$data             = $this->getItem();
			$data->catid      = $this->getState('ticket.catid', null) ?: $data->catid;
			$data->created_by = $data->created_by ?? Permissions::getUser()->id;
		}

		// Make sure data is an array
		$data = is_object($data) ? $data->getProperties() : $data;

		// Special considerations for new tickets
		$this->processNewTicketData($data);

		$this->preprocessData('com_ats.ticket', $data);

		return $data;
	}

	/** @inheritdoc */
	protected function prepareTable($table)
	{
		// Set up the created / modified date
		$date  = Factory::getDate();
		$user  = Factory::getApplication()->getIdentity();
		$isNew = empty($table->getId());

		if ($isNew)
		{
			// Set the values
			$table->created = $date->toSql();

			if (!TicketTable::$overrideuser)
			{
				$table->created_by = $user->id;
			}
		}
		else
		{
			/**
			 * We will NOT set the modified fields on existing tickets. These are used to indicate who and when last
			 * replied to a ticket.
			 */
			// $table->modified    = $date->toSql();
			// $table->modified_by = $user->id;
		}
	}

	/**
	 * Get the category parameters
	 *
	 * @param   int|null  $catid  The category ID
	 *
	 * @return  Registry  The category parameters
	 *
	 * @since   5.0.0
	 */
	private function getCatParams(?int $catid): Registry
	{
		if (empty($catid))
		{
			return new Registry();
		}

		$catService = new Category([]);
		$category   = $catService->get($catid);

		if (empty($category))
		{
			return new Registry();
		}

		return new Registry($category->params);
	}

	private function isNewTicket(Form $form): bool
	{
		$ticketId = $form->getValue('id');

		return empty($ticketId) || ((int) $ticketId <= 0);
	}

	/**
	 * Migrate old, ATS 1.x through 4.x custom field data into Joomla native custom fields.
	 *
	 * @param   string|null  $params  The raw content of the ticket's params field
	 * @param   Form         $form    The form we are manipulating
	 * @param   object|null  $item    The currently loaded ticket item
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function migrateLegacyCustomFieldsData(?string $params, Form $form, ?object $item): void
	{
		// Get legacy ATS custom field data from the parameters
		$legacyData = @json_decode($params, true) ?? [];

		// Get all currently defined custom fields
		$fields           = $form->getGroup('com_fields');
		$customFieldNames = array_map(
			function (FormField $field) {
				return $field->fieldname;
			}, $fields
		);

		// Only keep the legacy custom field data that matches the current fields' names
		$legacyData = array_filter(
			$legacyData, function (string $key) use ($customFieldNames) {
			return in_array($key, $customFieldNames);
		}, ARRAY_FILTER_USE_KEY
		);

		if (empty($legacyData))
		{
			// No data to migrate.
			return;
		}

		// Get the current custom field data
		$customFieldData = (array) $form->getData()->get('com_fields', (object) []);

		// Remove any legacy data if the same–named custom field already has a value
		$legacyData = array_filter(
			$legacyData, function (string $key) use ($customFieldData) {
			return !array_key_exists($key, $customFieldData);
		}, ARRAY_FILTER_USE_KEY
		);

		if (empty($legacyData))
		{
			// No data to migrate.
			return;
		}

		// Migrate leftover $legacyData into the database
		$fields = FieldsHelper::getFields('com_ats.ticket', $item);

		foreach ($fields as $field)
		{
			if (!array_key_exists($field->name, $legacyData))
			{
				continue;
			}

			$value = $legacyData[$field->name];

			// If no value set (empty) remove value from database
			if (is_array($value) ? !count($value) : !strlen($value))
			{
				$value = null;
			}

			// JSON encode value for complex fields
			if (is_array($value)
			    && (count($value, COUNT_NORMAL) !== count($value, COUNT_RECURSIVE)
			        || !count(
						array_filter(array_keys($value), 'is_numeric')
					)))
			{
				$value = json_encode($value);
			}

			/**
			 * Setting the value for the field and the item.
			 *
			 * I cannot go through \Joomla\Component\Fields\Administrator\Model\FieldModel::setFieldValue() because it
			 * does an ACL check **on the current user**. I need to migrate the data regardless of the current user.
			 * If a public ticket is accessed by a guest I still need to migrate its data. Therefore, I need to reinvent
			 * the wheel, in typical Joomla fashion. DAMN.
			 */
			$this->setFieldValue($field->id, $item->id, $value);
		}

		// Load the new data into the form (this only applies for edit forms, not for front–end display)
		foreach ($legacyData as $fieldName => $fieldValue)
		{
			$form->setValue($fieldName, 'com_fields', $fieldValue);
		}

		/**
		 * Finally, clear the custom fields helper, forcing it to reload the custom field data values. This is necessary
		 * for ticket display, not for the edit form.
		 */
		FieldsHelper::clearFieldsCache();
	}

	/**
	 * Special form data processing for new tickets
	 *
	 * @param   array  $data
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	private function processNewTicketData(array &$data): void
	{
		$ticketId  = $data['id'] ?? null;
		$newTicket = empty($ticketId) || ((int) $ticketId <= 0);
		$catid     = $data['catid'] ?? null;

		if (!$newTicket || empty($catid))
		{
			return;
		}

		$catService = new Category([]);
		$category   = $catService->get($catid);
		$catParams  = new Registry($category->params);
		$cParams    = ComponentHelper::getParams('com_ats');

		$postRaw = trim(
			strip_tags(html_entity_decode($data['content_html'] ?? '')),
			' \t\n\r\0\x0B   ' . html_entity_decode('&nbsp;')
		);

		$forceType       = $catParams->get('forcetype', '');
		$defaultPrivate  = $catParams->get('defaultprivate', 0) == 1;
		$defaultPriority = $catParams->get('default_priority', '');
		$defaultPost     = $catParams->get('defposttext', '');
		$hasPriorities   = $cParams->get('ticketPriorities', 0) == 1;

		if ($forceType)
		{
			$data['public'] = (strtoupper($forceType) == 'PUB') ? 1 : 0;
		}
		elseif (($defaultPrivate) && !is_numeric(($data['public'] ?? null)))
		{
			$data['public'] = 0;
		}

		if ($hasPriorities && is_numeric($defaultPriority)
		    && (!isset($data['priority'])
		        || !is_numeric(
					$data['priority']
				)))
		{
			$data['priority'] = (int) $defaultPriority;
		}

		if (empty($postRaw) && !empty($defaultPost))
		{
			$data['content_html'] = $defaultPost;
		}
	}

	/**
	 * Special processing of the form fields for new tickets.
	 *
	 * Some form fields need to be remove depending on the user privileges and the category options.
	 *
	 * @param   Form  $form  The form to post–process
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function processNewTicketForm(Form $form): void
	{
		$catid = $form->getValue('catid') ?? null;

		if (empty($catid))
		{
			foreach ($form->getFieldsets() as $fieldset => $fieldsetInfo)
			{
				foreach ($form->getFieldset($fieldset) as $field)
				{
					if ($field->fieldname == 'catid')
					{
						continue;
					}

					$form->removeField($field->fieldname);
				}
			}
		}

		$catParams = $this->getCatParams($catid);
		$cParams   = ComponentHelper::getParams('com_ats');
		$user      = Permissions::getUser();

		$timeSpentHidden    = $cParams->get('timespent_hide', 0) == 1;
		$timeSpentMandatory = !$timeSpentHidden && $cParams->get('timespent_mandatory', 0) != 1;

		$isGuest   = $user->guest;
		$isPro     = defined('ATS_PRO') && ATS_PRO;
		$isManager = Permissions::isManager($catid);
		$canPost   = $user->authorise('core.create', 'com_ats')
		             || $user->authorise(
				'core.create', 'com_ats.category.' . $catid
			);
		$canAttach = $user->authorise('ats.attachment', 'com_ats')
		             || $user->authorise(
				'ats.attachment', 'com_ats.category.' . $catid
			);
		$canAttach = $canAttach && $isPro;

		$forceType     = $catParams->get('forcetype', '');
		$hasPriorities = $cParams->get('ticketPriorities', 0) == 1;

		$fieldChanges = [
			'name'         => $isGuest ? '' : 'remove',
			'username'     => $isGuest ? '' : 'remove',
			'email'        => $isGuest ? '' : 'remove',
			'catid'        => $this->isFrontendNewTicketForCategory() ? 'remove' : '',
			'captcha'      => $isGuest ? '' : 'remove',
			'created_by'   => $isManager ? '' : 'remove',
			'public'       => empty($forceType) ? '' : 'disable',
			'priority'     => $hasPriorities ? '' : 'remove',
			'content_html' => $canPost ? '' : 'remove',
			'timespent'    => !$timeSpentHidden && $isManager ? '' : 'remove',
			'attachments'  => $canAttach ? '' : 'remove',
			'usertags'     => ($isManager && $isPro) ? '' : 'remove',
		];

		// The category field on multilanguage sites needs to be filtered by language for new tickets
		if (Multilanguage::isEnabled())
		{
			$app = Factory::getApplication();

			$form->setFieldAttribute('catid', 'language', sprintf("*,%s", $app->getLanguage()->getTag()));
		}

		// Populate the default value of the created_by field
		$created_by = $form->getValue('created_by', null, $user->id);
		$form->setValue('created_by', null, $created_by);

		array_walk(
			$fieldChanges, function (string $type, string $fieldName, Form $form) {
			switch ($type)
			{
				case 'disable':
					$form->setFieldAttribute($fieldName, 'disabled', 'true');
					$form->setFieldAttribute($fieldName, 'readonly', 'true');
					$form->setFieldAttribute($fieldName, 'required', 'false');
					$form->setFieldAttribute($fieldName, 'filter', 'unset');
					$form->setFieldAttribute($fieldName, 'layout', '');
					$form->setFieldAttribute($fieldName, 'description', '');
					break;

				case 'remove':
					$form->removeField($fieldName);
					break;
			}
		}, $form
		);

		// Is timespent a required field?
		if (!$timeSpentMandatory)
		{
			$form->setFieldAttribute('timespent', 'required', 'false');
		}
	}

	/**
	 * Save the value of a custom field.
	 *
	 * Used to migrate old custom fields. Based on FieldModel::setFieldValue(), albeit without an ACL check on the
	 * current user.
	 *
	 * @param   int    $fieldId  The Joomla custom field ID
	 * @param   int    $itemId   The ticket ID we are currently editing
	 * @param   mixed  $value    The value to set the custom field to
	 *
	 * @return  void
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 *
	 * @see     \Joomla\Component\Fields\Administrator\Model\FieldModel::setFieldValue()
	 */
	private function setFieldValue(int $fieldId, int $itemId, $value): void
	{
		/** @var FieldModel $model */
		$model = Factory::getApplication()->bootComponent('com_fields')->getMVCFactory()
			->createModel('Field', 'Administrator', ['ignore_request' => true]);

		$needsDelete = false;
		$needsInsert = false;
		$needsUpdate = false;

		$oldValue = $model->getFieldValue($fieldId, $itemId);
		$value    = (array) $value;

		if ($oldValue === null)
		{
			// No records available, doing normal insert
			$needsInsert = true;
		}
		elseif (count($value) == 1 && count((array) $oldValue) == 1)
		{
			// Only a single row value update can be done when not empty
			$needsUpdate = is_array($value[0]) ? count($value[0]) : strlen($value[0]);
			$needsDelete = !$needsUpdate;
		}
		else
		{
			// Multiple values, we need to purge the data and do a new insert
			$needsDelete = true;
			$needsInsert = true;
		}

		$db = $this->getDatabase();

		if ($needsDelete)
		{
			// Deleting the existing record as it is a reset
			$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true));

			$query->delete($query->quoteName('#__fields_values'))
				->where($query->quoteName('field_id') . ' = :fieldid')
				->where($query->quoteName('item_id') . ' = :itemid')
				->bind(':fieldid', $fieldId, ParameterType::INTEGER)
				->bind(':itemid', $itemId);

			$db->setQuery($query)->execute();
		}

		if ($needsInsert)
		{
			$newObj = new stdClass();

			$newObj->field_id = $fieldId;
			$newObj->item_id  = $itemId;

			foreach ($value as $v)
			{
				$newObj->value = $v;

				$db->insertObject('#__fields_values', $newObj);
			}
		}

		if ($needsUpdate)
		{
			$updateObj = new stdClass();

			$updateObj->field_id = $fieldId;
			$updateObj->item_id  = $itemId;
			$updateObj->value    = reset($value);

			$db->updateObject('#__fields_values', $updateObj, ['field_id', 'item_id']);
		}
	}
}
