<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Model;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Mixin\ModelPopulateStateTrait;
use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Database\ParameterType;
use Joomla\Database\QueryInterface;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;

#[\AllowDynamicProperties]
class TicketsModel extends ListModel
{
	use ModelPopulateStateTrait;

	public function __construct($config = [], ?MVCFactoryInterface $factory = null)
	{
		$config['filter_fields'] = $config['filter_fields'] ?? [
				// Sortable columns (also overlaps with filter columns)
				'id', 'catid', 'status', 'title', 'alias', 'public', 'priority', 'origin', 'assigned_to', 'created',
				'created_by', 'modified', 'modified_by', 'enabled', 'language',
				// Sort–only fields
				't.id', 't.status', 'title', 'created_name', 'cat_title', 'language_title',
				// Filter–only fields
				'search', 'user', 'since', 'to', 'tag',
			];

		parent::__construct($config, $factory);

		$this->setupStateFilters([
			'search'      => 'string',
			'user'        => 'string',
			'created_by'  => 'int',
			'assigned_to' => 'int',
			'catid'       => 'array',
//			'status'      => 'array',
			'status'      => 'string',
			'public'      => 'int',
			'priority'    => 'int',
			'origin'      => 'string',
			'enabled'     => 'int',
			'since'       => 'string',
			'to'          => 'string',
			'tag'         => 'ignore',
			'access'      => 'ignore',
			'language'    => 'ignore',
		], 'modified', 'DESC');
	}

    public function hasInvitedUsers($items, $item)
    {
        static $cache = null;

        if (is_null($cache))
        {
            $id_tickets = [];

            foreach ($items as $ticket)
            {
                $id_tickets[] = $ticket->id;
            }

            $db = $this->getDatabase();
            $query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
                        ->select($db->quoteName('ticket_id'))
                        ->from($db->quoteName('#__ats_tickets_users'))
                        ->whereIn('ticket_id', $id_tickets);

            $cache = $db->setQuery($query)->loadColumn();
        }

        return in_array($item->id, $cache);
    }

	protected function getListQuery()
	{
		$db    = $this->getDatabase();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select([
				$db->quoteName('t') . '.*',
				$db->quoteName('c.language'),
				$db->quoteName('c.title', 'cat_title'),
				$db->quoteName('c.alias', 'cat_alias'),
				$db->quoteName('c.path', 'cat_path'),
				$db->quoteName('c.published', 'cat_published'),
				$db->quoteName('c.params', 'cat_params'),
				$db->quoteName('uc.name', 'created_name'),
				$db->quoteName('l.title', 'language_title'),
				$db->quoteName('l.image', 'language_image'),
			])
			->from($db->quoteName('#__ats_tickets', 't'))
			->join(
				'INNER',
				$db->quoteName('#__categories', 'c'),
				$db->quoteName('c.id') . ' = ' . $db->quoteName('t.catid')
			)
			->join(
				'LEFT',
				$db->quoteName('#__languages', 'l'), $db->quoteName('l.lang_code') . ' = ' . $db->quoteName('c.language')
			)
			->join(
				'LEFT',
				$db->quoteName('#__users', 'uc'),
				$db->quoteName('uc.id') . ' = ' . $db->quoteName('t.created_by')
			)
			->where($db->quoteName('c.extension') . ' = ' . $db->quote('com_ats'));

		// FILTER: Access Levels
		$this->ensureAccessLevel();
		$accessLevels = $this->getState('filter.access', []);
		if (!empty($accessLevels))
		{
			$query->whereIn($db->quoteName('c.access'), $accessLevels, ParameterType::INTEGER);
		}

		// FILTER: Language
		$this->ensureLanguage();
		$filterLanguage = $this->getState('filter.language', ['*']);
		$filterLanguage = ((empty($filterLanguage)) || ($filterLanguage === ['*'])) ? null : $filterLanguage;

		if (is_array($filterLanguage))
		{
			$query->whereIn($db->quoteName('c.language'), $filterLanguage, ParameterType::STRING);
		}

		// FILTER: search (support prefixes id, title, category)
		$fltSearch = $this->getState('filter.search');

		if (!empty($fltSearch))
		{
			if (stripos($fltSearch, 'id:') === 0)
			{
				$ids = (int) substr($fltSearch, 3);
				$query->where($db->quoteName('t.id') . ' = :id')
					->bind(':id', $ids, ParameterType::INTEGER);
			}
			elseif (stripos($fltSearch, 'category:') === 0)
			{
				$fltSearch = '%' . substr($fltSearch, 9) . '%';
				$query->where($db->quoteName('c.title') . ' LIKE :search')
					->bind(':search', $fltSearch, ParameterType::STRING);
			}
			else
			{
				if (stripos($fltSearch, 'title:') === 0)
				{
					$fltSearch = substr($fltSearch, 6);
				}
				$fltSearch = '%' . $fltSearch . '%';
				$query->where($db->quoteName('t.title') . ' LIKE :search')
					->bind(':search', $fltSearch, ParameterType::STRING);
			}
		}

		// FILTER: user (prefixes: username, name, email, id) / created_by
		$fltUser      = $this->getState('filter.user');
		$fltCreatedBy = $this->getState('filter.created_by');
		$fltUser      = (!is_null($fltCreatedBy) && ($fltCreatedBy > 0)) ? ('id:' . (int) $fltCreatedBy) : $fltUser;

		if (!empty($fltUser))
		{
			if (stripos($fltUser, 'id:') === 0)
			{
				$ids = (int) substr($fltUser, 3);
				$query->where($db->quoteName('t.created_by') . ' = :id')
					->bind(':id', $ids, ParameterType::INTEGER);
			}
			else
			{
				if (stripos($fltUser, 'username:') === 0)
				{
					$username = '%' . substr($fltUser, 9) . '%';
					$query
						->where($db->quoteName('uc.username') . ' LIKE :username')
						->bind(':username', $username);
				}
				elseif (stripos($fltUser, 'name:') === 0)
				{
					$name = '%' . substr($fltUser, 5) . '%';
					$query
						->where($db->quoteName('uc.name') . ' LIKE :name')
						->bind(':name', $name);
				}
				elseif (stripos($fltUser, 'email:') === 0)
				{
					$email = '%' . substr($fltUser, 6) . '%';
					$query
						->where($db->quoteName('uc.email') . ' LIKE :email')
						->bind(':email', $email);
				}
                elseif (stripos($fltUser, 'invited:') === 0)
                {
                    $username = '%' . substr($fltUser, 8) . '%';
                    $name     = $username;
                    $email    = $username;

	                /**
	                 * On MySQL-compatible databases I can use a WHERE EXISTS subquery.
	                 * @link https://dev.mysql.com/doc/refman/8.0/en/exists-and-not-exists-subqueries.html
	                 */
					if ($db->getServerType() === 'mysql')
					{
						/** @var QueryInterface $subQuery */
						$subQuery = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true));
						$subQuery->select(1)
							->from($db->quoteName('#__ats_tickets_users', 'iut'))
							->join(
								'INNER',
								$db->quoteName('#__users', 'iuj'),
								$db->quoteName('iuj.id') . ' = ' . $db->quoteName('iut.user_id')
							)
							->where($db->quoteName('iut.ticket_id') . ' = ' . $db->quoteName('t.id'))
							->extendWhere(
								'AND',
								[
									$db->quoteName('iuj.username') . ' LIKE :invited_specifier_1',
									$db->quoteName('iuj.name') . ' LIKE :invited_specifier_2',
									$db->quoteName('iuj.email') . ' LIKE :invited_specifier_3',
								],
								'OR'
							);
						$query->where('EXISTS(' . $subQuery . ')')
							->bind(':invited_specifier_1', $username)
							->bind(':invited_specifier_2', $name)
							->bind(':invited_specifier_3', $email);
					}
					else
					{
	                    // Let's get the user id and then use it for invited users.
	                    $user_query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
	                        ->select($db->quoteName('id'))
	                        ->from($db->quoteName('#__users'))
	                        ->where([
	                            $db->quoteName('username') . ' LIKE :username',
	                            $db->quoteName('name') . ' LIKE :name',
	                            $db->quoteName('email') . ' LIKE :email',
	                        ], 'OR')
	                        ->bind(':username', $username)
	                        ->bind(':name', $name)
	                        ->bind(':email', $email);

	                    $user_ids = $db->setQuery($user_query)->loadColumn() ?: [-1];

	                    // Now let's get the tickets where the invited users fulfills any of the above criteria
	                    $ticket_query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
	                        ->select($db->quoteName('ticket_id'))
	                        ->from($db->quoteName('#__ats_tickets_users'))
	                        ->whereIn('user_id', $user_ids);
	                    $ticket_ids = $db->setQuery($ticket_query)->loadColumn();
	                    $ticket_ids = implode(', ', $ticket_ids);

	                    $query
	                        ->extendWhere('AND', [
	                            $db->quoteName('t.id'). ' IN(:ticket_ids)',
	                        ], 'OR')
	                        ->bind(':ticket_ids', $ticket_ids);
					}
                }
				else
				{
					$username = '%' . $fltUser . '%';
					$name     = $username;
					$email    = $username;

					$query
						->extendWhere('AND', [
							$db->quoteName('uc.username') . ' LIKE :username',
							$db->quoteName('uc.name') . ' LIKE :name',
							$db->quoteName('uc.email') . ' LIKE :email',
						], 'OR')
						->bind(':username', $username)
						->bind(':name', $name)
						->bind(':email', $email)
                    ;
				}
			}
		}

		// FILTER: assigned_to
		$fltAssignedTo = $this->getState('filter.assigned_to');

		if (is_numeric($fltAssignedTo))
		{
			$fltAssignedTo = (int) $fltAssignedTo;
			$query
				->where($db->quoteName('t.assigned_to') . ' = :assigned_to')
				->bind(':assigned_to', $fltAssignedTo, ParameterType::INTEGER);
		}

		// FILTER: catid
		$fltCatId = $this->getState('filter.catid') ?: [];
		$fltCatId = is_numeric($fltCatId) ? [$fltCatId] : $fltCatId;
		$fltCatId = array_filter($fltCatId, function ($x) {
			return !empty($x) && is_numeric($x) && ((int) $x > 0);
		});

		if (!empty($fltCatId))
		{
			$query->whereIn($db->quoteName('catid'), $fltCatId);
		}

		// FILTER: status
		$fltStatus = $this->getState('filter.status');

		if (!empty($fltStatus))
		{
			$query
				//->whereIn($db->quoteName('t.status'), $fltStatus);
				->where($db->quoteName('t.status') . ' = :status')
				->bind(':status', $fltStatus, ParameterType::STRING);
		}

		// FILTER: public
		$fltPublic = $this->getState('filter.public');

		if (is_numeric($fltPublic))
		{
			$fltPublic = (int) $fltPublic;
			$query
				->where($db->quoteName('t.public') . ' = :public')
				->bind(':public', $fltPublic, ParameterType::STRING);
		}

		// FILTER: priority
		$fltPriority = $this->getState('filter.priority');

		if (is_numeric($fltPriority))
		{
			$fltPriority = (int) $fltPriority;
			$query
				->where($db->quoteName('t.priority') . ' = :priority')
				->bind(':priority', $fltPriority, ParameterType::STRING);
		}

		// FILTER: origin
		$fltOrigin = $this->getState('filter.origin');

		if (!empty($fltOrigin))
		{
			$query
				->where($db->quoteName('t.origin') . ' = :origin')
				->bind(':origin', $fltOrigin, ParameterType::STRING);
		}

		// FILTER: enabled
		$fltEnabled = $this->getState('filter.enabled');

		if (is_numeric($fltEnabled))
		{
			$fltEnabled = (int) $fltEnabled;
			$query
				->where($db->quoteName('t.enabled') . ' = :enabled')
				->bind(':enabled', $fltEnabled, ParameterType::STRING);
		}

		// FILTER: since / to
		$fltSince = $this->getState('filter.since', null);
		$fltTo    = $this->getState('filter.to', null);
		try
		{
			$fltSince = empty($fltSince) ? null : clone Factory::getDate($fltSince);
		}
		catch (Exception $e)
		{
			$fltSince = null;
		}
		try
		{
			$fltTo = empty($fltTo) ? null : clone Factory::getDate($fltTo);
		}
		catch (Exception $e)
		{
			$fltTo = null;
		}

		if (!empty($fltSince) && empty($fltTo))
		{
			$since = $fltSince->toSql();
			$query
				->where($db->quoteName('t.created') . ' >= :since')
				->bind(':since', $since);
		}
		elseif (empty($fltSince) && !empty($fltTo))
		{
			$to = $fltTo->toSql();
			$query
				->where($db->quoteName('t.created') . ' <= :to')
				->bind(':to', $to);
		}
		elseif (!empty($fltSince) && !empty($fltTo))
		{
			$since = $fltSince->toSql();
			$to    = $fltTo->toSql();

			if ($fltSince->diff($fltTo)->invert)
			{
				$temp  = $to;
				$to    = $since;
				$since = $to;
				unset($temp);
			}

			$query
				->where($db->quoteName('t.created') . ' >= :since')
				->where($db->quoteName('t.created') . ' <= :to')
				->bind(':since', $since)
				->bind(':to', $to);
		}

		// FILTER: Tags
		$tag = $this->getState('filter.tag');

		if (!empty($tag))
		{
			// Run simplified query when filtering by one tag.
			if (\is_array($tag) && \count($tag) === 1)
			{
				$tag = $tag[0];
			}

			if (\is_array($tag))
			{
				$tag = ArrayHelper::toInteger($tag);

				$subQuery = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
					->select('DISTINCT ' . $db->quoteName('content_item_id'))
					->from($db->quoteName('#__contentitem_tag_map'))
					->whereIn($db->quoteName('tag_id'), $tag)
					->where($db->quoteName('type_alias') . ' = ' . $db->quote('com_ats.ticket'));

				$query->join(
					'INNER',
					'(' . $subQuery . ') AS ' . $db->quoteName('tagmap'),
					$db->quoteName('tagmap.content_item_id') . ' = ' . $db->quoteName('t.id')
				);
			}
			else
			{
				$tag = (int) $tag;
				$query->join(
					'INNER',
					$db->quoteName('#__contentitem_tag_map', 'tagmap'),
					$db->quoteName('tagmap.content_item_id') . ' = ' . $db->quoteName('t.id')
				)
					->where($db->quoteName('tagmap.type_alias') . ' = ' . $db->quote('com_ats.ticket'))
					->where($db->quoteName('tag_id') . '= :tag')
					->bind(':tag', $tag, ParameterType::INTEGER);
			}
		}

		// List ordering clause
		$orderCol  = $this->state->get('list.ordering', 'modified');
		$orderDirn = $this->state->get('list.direction', 'DESC');
		$ordering  = $db->escape($orderCol) . ' ' . $db->escape($orderDirn);

		$query->order($ordering);

		return $query;
	}

	/**
	 * Ensure that filter.access is non–empty.
	 *
	 * If it's empty we will populate it with the currently logged in user's authorized access levels.
	 *
	 * @throws Exception
	 * @since  5.0.0
	 */
	private function ensureAccessLevel()
	{
		$accessLevel = $this->getState('filter.access', null) ?: [];

		if (!empty($accessLevel) && !is_array($accessLevel))
		{
			$accessLevel = explode(',', $accessLevel);
			$accessLevel = array_map(function ($x) {
				return is_numeric($x) ? ((int) $x) : null;
			}, $accessLevel);
			$accessLevel = array_filter($accessLevel, function ($x) {
				return !empty($x) && ($x > 0);
			});
		}

		if (empty($accessLevel))
		{
			$user        = Factory::getApplication()->getIdentity();
			$accessLevel = $user->getAuthorisedViewLevels();
		}

		$this->setState('filter.access', $accessLevel);
	}

	/**
	 * Make sure that filter.language is populated
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function ensureLanguage(): void
	{
		$language = $this->getState('filter.language', null);

		if (!empty($language))
		{
			if (is_string($language))
			{
				$language = explode(',', $language);
				$language = array_map('trim', $language);
			}

			$this->setState('filter.language', $language);

			return;
		}

		$app             = Factory::getApplication();
		$defaultLanguage = ['*'];

		if ($app->isClient('site'))
		{
			$hasLanguageFilter = method_exists($app, 'getLanguageFilter') && $app->getLanguageFilter();
			$defaultLanguage   = $hasLanguageFilter ? $this->getCurrentLanguages(true) : $defaultLanguage;
		}

		$this->setState('filter.language', $defaultLanguage);
	}

	/**
	 * Get the currently applicable languages.
	 *
	 * @param   bool  $includeDefault  Should I include the default “all languages” item as well?
	 *
	 * @return  array|string[]
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function getCurrentLanguages(bool $includeDefault = true): array
	{
		$languages = $includeDefault ? ['*'] : [];
		$plugin    = PluginHelper::getPlugin('system', 'languagefilter');

		if (!is_object($plugin) || !property_exists($plugin, 'params'))
		{
			return $languages;
		}

		$languages[] = (new Registry($plugin->params))->get('remove_default_prefix')
			? Factory::getApplication()->getLanguage()->getTag()
			: Factory::getApplication()->getInput()->getCmd('language', '*');

		// Filter out double languages
		return array_unique($languages);
	}
}