<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Model\UpgradeHandler;

use Akeeba\Component\ATS\Administrator\Helper\BBCode;
use Akeeba\Component\ATS\Administrator\Helper\UserTags;
use Akeeba\Component\ATS\Administrator\Model\UpgradeModel;
use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\Filesystem\Path;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\Component\Tags\Administrator\Table\TagTable;
use Joomla\Database\DatabaseDriver;
use Joomla\Database\ParameterType;
use Joomla\Registry\Registry;
use ReflectionClass;
use Throwable;

defined('_JEXEC') or die;

/**
 * Custom UpgradeModel handler for migrating component options from Akeeba Ticket System 4 to 5+
 *
 * @since  5.0.0
 */
class MigrateSettings
{
	/**
	 * Joomla database driver object
	 *
	 * @var   DatabaseDriver
	 * @since 5.0.0
	 */
	private $dbo;

	/**
	 * The UpgradeModel instance we belong to.
	 *
	 * @var   UpgradeModel
	 * @since 5.0.0
	 */
	private $upgradeModel;

	/**
	 * Constructor.
	 *
	 * @param   UpgradeModel  $upgradeModel  The UpgradeModel instance we belong to
	 *
	 * @since   5.0.0
	 */
	public function __construct(UpgradeModel $upgradeModel, DatabaseDriver $dbo)
	{
		$this->upgradeModel = $upgradeModel;
		$this->dbo          = $dbo;
	}

	/**
	 * Runs after updating the component
	 *
	 * @return  bool
	 *
	 * @since   5.0.0
	 */
	public function onUpdate(): bool
	{
		$this->runIsolated(
			[
				'migrateAttachmentsFolder',
				'migrateCannedReplies',
				'migrateShowGuestCaptcha',
				'migrateUserTags',
				'migrateCustomStatus',
			]
		);

		return true;
	}

	/**
	 * Get the legacy ATS User Tags.
	 *
	 * Will only return anything when updating from ATS 4 or earlier.
	 *
	 * @return  array|mixed
	 *
	 * @since   5.0.0
	 */
	private function getAtsUserTags(): array
	{
		$db    = $this->dbo;
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select(
				[
					$db->quoteName('ats_usertag_id'),
					$db->quoteName('title'),
					$db->quoteName('descr'),
				]
			)
			->from($db->quoteName('#__ats_usertags'));
		try
		{
			return $db->setQuery($query)->loadAssocList() ?: [];
		}
		catch (Exception $e)
		{
			return [];
		}
	}

	/**
	 * Map an arbitrary Akeeba Ticket System user tag.
	 *
	 * Returns the ID of a core Joomla tag with the same title.
	 *
	 * If a tag with the same title does not already exist a new one will be created, unpublished and assigned to
	 * Joomla's default Special access level.
	 *
	 * @param   string  $title        The tag title we will search for
	 * @param   string  $description  Optional description to use if we have to create the tag.
	 *
	 * @return  int|null  The new tag ID. NULL if there is no tag and one cannot be created.
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function mapArbitraryTag(string $title, string $description = ''): ?int
	{
		$db    = $this->dbo;
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select(
				[
					$db->quoteName('id'),
				]
			)
			->where($db->quoteName('title') . ' = :title')
			->bind(':title', $title);
		$newId = $db->setQuery($query)->loadResult() ?: null;

		if (!empty($newId) && ($newId > 1))
		{
			return $newId;
		}

		/** @var MVCFactoryInterface $factory */
		$factory = Factory::getApplication()->bootComponent('com_tags')
			->getMVCFactory();
		/** @var TagTable $tagTable */
		$tagTable = $factory->createTable('Tag', 'Administrator');

		try
		{
			$saved = $tagTable->save(
				[
					'parent_id'   => $tagTable->getRootId(),
					'title'       => $title,
					'description' => $description,
					'note'        => 'Automatically migrated Akeeba Ticket System user tag. Please make sure it is only visible to Support Staff before publishing.',
					'published'   => 0,
					'access'      => 3,
					'language'    => '*',
				]
			);
		}
		catch (Exception $e)
		{
			return null;
		}

		if (!$saved)
		{
			return null;
		}

		return $tagTable->getId();
	}

	/**
	 * Migrate the attachments_folder option
	 *
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function migrateAttachmentsFolder(): void
	{
		// Get the configured attachments folder
		$cParams     = ComponentHelper::getParams('com_ats');
		$hasMigrated = $cParams->get('migrate_media_options', 0);

		if ($hasMigrated)
		{
			return;
		}

		$cParams->set('migrate_media_options', 1);

		$directory = $cParams->get('attachments_folder', 'media/com_ats/attachments') ?: '';
		$directory = trim($directory);
		$directory = rtrim($directory, '/\\');

		// If no folder is set use the default
		if (empty($directory))
		{
			$cParams->set('attachments_folder', 'media/com_ats/attachments');
			$this->saveParameters($cParams);

			return;
		}

		/**
		 * ATS 1 through 4 would accept an attachment path as one of the following three cases:
		 *
		 * - As a subdirectory of your site's media folder.
		 * - As a subdirectory of your site's installation root.
		 * - As an absolute path on your server.
		 *
		 * The new version only accepts a subdirectory of your site's installation root. We can migrate the first legacy
		 * case, accept the second legacy case (identical to the current one) and replace the third legacy case with the
		 * default attachments folder.
		 */

		// Is this an existing media subdirectory (legacy)?
		$asMediaSubfolder = JPATH_ROOT . '/media/' . $directory;

		try
		{
			$cleanPath = Path::check($asMediaSubfolder);

			if (@is_dir($cleanPath))
			{
				// Directory points to an existing media subfolder. Update the value.
				$cParams->set('attachments_folder', 'media/' . $directory);
				$this->saveParameters($cParams);

				return;
			}

		}
		catch (Exception $e)
		{
			// Relative path (double dots) or out of bounds path detected.
		}

		// Is this an absolute path pointing to a folder under the site's root (legacy)?
		$asAbsolutePath = $directory;

		try
		{
			$cleanPath    = Path::check($asAbsolutePath);
			$newDirectory = (JPATH_ROOT != '') ? trim(
				substr($cleanPath, 0, strlen(JPATH_ROOT)), '/' . DIRECTORY_SEPARATOR
			) : $cleanPath;

			if ($directory != $newDirectory)
			{
				$cParams->set('attachments_folder', $newDirectory);
				$this->saveParameters($cParams);
			}

			return;
		}
		catch (Exception $e)
		{
			// Relative path (double dots) or out of bounds path detected.
		}

		// Is this a relative path pointing to a folder under the site's root (legacy or current)?
		$asSiteFolder = JPATH_ROOT . '/' . $directory;

		try
		{
			$cleanPath    = Path::check($asSiteFolder);
			$newDirectory = (JPATH_ROOT != '') ? trim(substr($cleanPath, strlen(JPATH_ROOT)), '/' . DIRECTORY_SEPARATOR)
				: $cleanPath;

			if ($directory != $newDirectory)
			{
				$cParams->set('attachments_folder', $newDirectory);
				$this->saveParameters($cParams);
			}

			return;
		}
		catch (Exception $e)
		{
			// Relative path (double dots) or out of bounds path detected.
		}

		// If you are still here this is an invalid folder. Use default.
		$cParams->set('attachments_folder', 'media/com_ats/attachments');
		$this->saveParameters($cParams);
	}

	/**
	 * Migrate BBcode canned replies to HTML content
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	private function migrateCannedReplies(): void
	{
		$cParams     = ComponentHelper::getParams('com_ats');
		$hasMigrated = $cParams->get('migrate_canned_replies', 0);

		if ($hasMigrated)
		{
			return;
		}

		$cParams->set('migrate_canned_replies', 1);
		$this->saveParameters($cParams);

		if (!class_exists(BBCode::class))
		{
			@include_once JPATH_ADMINISTRATOR . '/components/com_ats/Helper/BBCode.php';
		}

		if (!class_exists(BBCode::class))
		{
			return;
		}

		$db    = $this->dbo;
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select(
				[
					$db->quoteName('id'),
					$db->quoteName('reply'),
				]
			)
			->from($db->quoteName('#__ats_cannedreplies'))
			->order($db->quoteName('id') . ' ASC');

		$limit = 0;

		while (true)
		{
			$oldLimit = $limit;
			$limit    += 50;

			$query->setLimit($oldLimit, $limit);

			try
			{
				$replies = $db->setQuery($query)->loadAssocList();
			}
			catch (Exception $e)
			{
				break;
			}

			if (empty($replies))
			{
				break;
			}

			foreach ($replies as $reply)
			{
				$id         = $reply['id'];
				$content    = $reply['reply'];
				$newContent = BBCode::parseBBCodeConditional($content);

				if ($newContent == $content)
				{
					continue;
				}

				$updateQuery = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
					->update($db->quoteName('#__ats_cannedreplies'))
					->set($db->quoteName('reply') . ' = :newContent')
					->where($db->quoteName('id') . ' = :id')
					->bind(':newContent', $newContent, ParameterType::STRING)
					->bind(':id', $id, ParameterType::INTEGER);
				try
				{
					$db->setQuery($updateQuery)->execute();
				}
				catch (Exception $e)
				{
					// Don't worry if it fails.
				}
			}
		}
	}

	/**
	 * Migrate the custom ticket statuses
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function migrateCustomStatus(): void
	{
		$cParams     = ComponentHelper::getParams('com_ats');
		$hasMigrated = $cParams->get('migrate_custom_status', 0);

		if ($hasMigrated)
		{
			return;
		}

		$cParams->set('migrate_custom_status', 1);

		$custom = $cParams->get('customStatuses', '');

		// If there are no custom statuses or already migrated / created by ATS 5 there is nothing more to do.
		if (empty($custom) || is_object($custom))
		{
			return;
		}

		$custom = str_replace("\\n", "\n", $custom);
		$custom = str_replace("\r", "\n", $custom);
		$custom = str_replace("\n\n", "\n", $custom);
		$lines  = explode("\n", $custom);

		$statuses = [];

		foreach ($lines as $line)
		{
			$parts = explode('=', $line);

			if (count($parts) != 2)
			{
				continue;
			}

			$parts[0] = trim($parts[0]);
			$parts[1] = trim($parts[1]);

			if (!is_numeric($parts[0]))
			{
				continue;
			}

			$statusId = (int) $parts[0];

			if (($statusId <= 0) || ($statusId > 99))
			{
				continue;
			}

			$description = $parts[1];

			if (empty($description))
			{
				continue;
			}

			$statuses[$statusId] = Text::_($description);
		}

		if (empty($statuses))
		{
			return;
		}

		$toSave  = new \stdClass();
		$counter = 9;

		foreach ($statuses as $id => $label)
		{
			$key          = '__field' . ++$counter;
			$toSave->$key = (object) [
				'id'    => $id,
				'label' => $label,
			];
		}

		$cParams->set('customStatuses', $toSave);

		$this->saveParameters($cParams);
	}

	/**
	 * Migrate the legacy showGuestCAPTCHA option to the new captcha option.
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function migrateShowGuestCaptcha(): void
	{
		$cParams     = ComponentHelper::getParams('com_ats');
		$hasMigrated = $cParams->get('migrate_show_guest_captcha', 0);

		if ($hasMigrated)
		{
			return;
		}

		$cParams->set('migrate_show_guest_captcha', 1);

		$legacyOption = $cParams->get('showGuestCAPTCHA', null);

		if (is_null($legacyOption))
		{
			return;
		}

		$newOption = ($legacyOption == 1) ? '' : '0';
		$cParams->set('captcha', $newOption);
		$cParams->set('showGuestCAPTCHA', null);

		$this->saveParameters($cParams);
	}

	/**
	 * Migrate ATS 1–4 user tags to Joomla core tags
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function migrateUserTags()
	{
		$cParams     = ComponentHelper::getParams('com_ats');
		$hasMigrated = $cParams->get('migrate_user_tags', 0);

		if ($hasMigrated)
		{
			return;
		}

		$cParams->set('migrate_user_tags', 1);
		$this->saveParameters($cParams);

		if (!class_exists(UserTags::class))
		{
			@include_once JPATH_ADMINISTRATOR . '/components/com_ats/Helper/UserTags.php';
		}

		if (!class_exists(UserTags::class))
		{
			return;
		}

		$usertags = $this->getAtsUserTags();

		if (empty($usertags))
		{
			return;
		}

		// Map ATS user tags to core tags
		$atsToCoreTagMap = [];

		foreach ($usertags as $oldEntry)
		{
			$atsToCoreTagMap[$oldEntry['ats_usertag_id']] = $this->mapArbitraryTag(
				$oldEntry['title'], $oldEntry['descr']
			);
		}

		// Convert ATS tagged users to Joomla tagged users in batches of 50 users
		$db          = $this->dbo;
		$userIdQuery = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('DISTINCT ' . $db->quoteName('user_id'))
			->from($db->quoteName('#__ats_users_usertags'))
			->order($db->quoteName('user_id') . ' ASC');
		$limitEnd    = 0;

		while (true)
		{
			// Get a batch of user IDs
			$limitStart = $limitEnd;
			$limitEnd   += 50;
			$userIdQuery->setLimit($limitStart, $limitEnd);

			try
			{
				$userIds = $db->setQuery($userIdQuery)->loadColumn() ?: 0;
			}
			catch (Exception $e)
			{
				break;
			}

			if (empty($userIds))
			{
				break;
			}

			// Get the user ID to ATS tag ID map for these users
			$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
				->select(
					[
						$db->quoteName('user_id'),
						$db->quoteName('ats_usertag_id', 'tag_id'),
					]
				)
				->from($db->quoteName('#__ats_users_usertags'))
				->whereIn($db->quoteName('user_id'), $userIds, ParameterType::INTEGER);

			try
			{
				$rawData = $db->setQuery($query)->loadAssocList();
			}
			catch (Exception $e)
			{
				continue;
			}

			// Create an array of user ID to tag IDs
			$userToTags = [];

			foreach ($rawData as $entry)
			{
				$userToTags[$entry['user_id']]   = $userToTags[$entry['user_id']] ?? [];
				$userToTags[$entry['user_id']][] = $entry['tag_id'];
			}

			unset($rawData);

			// Map the ATS tag IDs to Joomla! tag IDs
			$userToTags = array_map(
				function (array $tags) use ($atsToCoreTagMap) {
					$tags = array_map(
						function ($tagId) use ($atsToCoreTagMap) {
							return $atsToCoreTagMap[$tagId] ?? null;
						}, $tags
					);

					$tags = array_filter(
						$tags, function ($x) {
						return !empty($x);
					}
					);

					return $tags;
				}, $userToTags
			);

			$userToTags = array_fill_keys(
				$userToTags, function ($x) {
				return !empty($x);
			}
			);

			if (empty($userToTags))
			{
				continue;
			}

			// Go through each user and tag them
			foreach ($userToTags as $userId => $tagIds)
			{
				UserTags::setUserTags($userId, $tagIds);
			}
		}
	}

	/**
	 * Runs a method inside a try/catch block to suppress any errors
	 *
	 * @param   string[]  $methodNames  The method name to run
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	private function runIsolated(array $methodNames): void
	{
		foreach ($methodNames as $methodName)
		{
			try
			{
				$this->{$methodName}();
			}
			catch (Throwable $e)
			{
				// No problem, let's move on.
			}
		}
	}

	/**
	 * Save the component parameters.
	 *
	 * We need to fork this code instead of using the ComponentParameters service because this class'
	 * methods will execute on extension upgrade, even when we are upgrading from ATS 4 which did not
	 * have a Joomla 4+ compatible namespace declaration in its XML file. As a result trying to boot
	 * the component to get the service will fail since Joomla does not have a PSR-4 autoloader set up
	 * for ATS yet.
	 *
	 * @param   Registry  $params
	 *
	 * @since   5.2.0
	 */
	private function saveParameters(Registry $params): void
	{
		$db   = $this->dbo;
		$data = $params->toString('JSON');

		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->update($db->qn('#__extensions'))
			->set($db->qn('params') . ' = ' . $db->quote($data))
			->where($db->qn('element') . ' = ' . $db->quote('com_ats'))
			->where($db->qn('type') . ' = ' . $db->quote('component'));

		$db->setQuery($query);

		try
		{
			$db->execute();
		}
		catch (\Exception $e)
		{
			// Don't sweat if it fails
		}

		// Reset ComponentHelper's cache
		$refClass = new ReflectionClass(ComponentHelper::class);
		$refProp  = $refClass->getProperty('components');

		if (version_compare(PHP_VERSION, '8.1.0', 'lt'))
		{
			$refProp->setAccessible(true);
		}

		if (version_compare(PHP_VERSION, '8.3.0', 'ge'))
		{
			$components = $refClass->getStaticPropertyValue('components');
		}
		else
		{
			$components = $refProp->getValue();
		}

		$components['com_ats']->params = $params;

		if (version_compare(PHP_VERSION, '8.3.0', 'ge'))
		{
			$refClass->setStaticPropertyValue('components', $components);
		}
		else
		{
			$refProp->setValue($components);
		}
	}

}