<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Table;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\Attachment;
use Akeeba\Component\ATS\Administrator\Mixin\TableAssertionTrait;
use Akeeba\Component\ATS\Administrator\Mixin\TableCreateModifyTrait;
use Joomla\Filesystem\File;
use Joomla\CMS\Language\Text;
use Joomla\Database\DatabaseDriver;
use Joomla\Event\DispatcherInterface;
use RuntimeException;

/**
 * Class AttachmentTable
 *
 * @property  int    $id                  Primary key
 * @property  int    $post_id             Link to the post containing the attachment
 * @property  string $original_filename   Original name of the attachment
 * @property  string $mangled_filename    Hashed filename
 * @property  string $mime_type           File mime-type
 * @property  string $origin              Attachment origin: web, email
 * @property  string $created             Date and time the attachment was created (UTC)
 * @property  int    $created_by          User ID who created this attachment
 * @property  int    $enabled             Is the attachment published?
 *
 * @since  5.0.0
 */
class AttachmentTable extends AbstractTable
{
	use TableCreateModifyTrait
	{
		TableCreateModifyTrait::onBeforeStore as onBeforeStoreCreateModifyAware;
	}
	use TableAssertionTrait;

	/**
	 * Files to delete when deleting an attachment, keyed by attachment ID
	 *
	 * @var   array
	 * @since 5.0.0
	 */
	private static $filesToDelete = [];

	/**
	 * The post this attachment belongs to
	 *
	 * @var   PostTable|null
	 * @since 5.0.0
	 */
	private $post = null;

	/**
	 * Public constructor
	 *
	 * @param   DatabaseDriver            $db          Database driver object to Joomla's DB
	 * @param   DispatcherInterface|null  $dispatcher  Joomla event dispatcher
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	public function __construct(DatabaseDriver $db, ?DispatcherInterface $dispatcher  = null)
	{
		$this->_supportNullValue = true;

		parent::__construct('#__ats_attachments', 'id', $db, $dispatcher);

		$this->setColumnAlias('published', 'enabled');
	}

	/**
	 * Get the absolute filename where an attachment is stored. NULL if there is no such file.
	 *
	 * @param   string|null  $mangledFilename  The (mangled) filename of the attachment. NULL to use the current
	 *                                         instance's mangled filename, if set.
	 *
	 * @return  string|null  The absolute path to the file. NULL if it doesn't exist or otherwise does not apply.
	 *
	 * @since   5.0.0
	 */
	public function getAbsoluteFilename(?string $mangledFilename = null): ?string
	{
		if (!class_exists(Attachment::class))
		{
			return null;
		}

		$mangledFilename = $mangledFilename ?? $this->mangled_filename;

		if (empty($mangledFilename))
		{
			return null;
		}

		$attachmentsDir = Attachment::getDirectory();

		if (empty($attachmentsDir))
		{
			return null;
		}

		$fileNames = [
			$attachmentsDir . '/' . substr($mangledFilename, 0, 2) . '/' . substr($mangledFilename, 2, 2) . '/' . $mangledFilename,
			$attachmentsDir . '/' . $mangledFilename,
		];

		foreach ($fileNames as $fileName)
		{
			if (!@file_exists($fileName) || !is_file($fileName))
			{
				continue;
			}

			return $fileName;
		}

		return null;
	}

	/**
	 * Get the post this attachment belongs to
	 *
	 * @return  PostTable|null
	 * @since   5.0.0
	 */
	public function getPost(): ?PostTable
	{
		if (is_null($this->post) && !empty($this->post_id))
		{
			$this->post = new PostTable($this->getDbo(), $this->getDispatcher());

			if (!$this->post->load($this->post_id))
			{
				$this->post = null;
			}
		}

		return $this->post;
	}

	/**
	 * Set the post this attachment belongs to
	 *
	 * @param   PostTable|null  $post   The post to assign
	 * @param   bool            $force  Should I force–assign the post?
	 *
	 * @return  AttachmentTable
	 *
	 * @since   5.0.0
	 */
	public function setPost(?PostTable $post, bool $force = false): AttachmentTable
	{
		if ($force)
		{
			$this->post    = $post;
			$this->post_id = empty($post) ? null : $post->getId();

			return $this;
		}

		if (!empty($post) && ($this->post_id != $post->getId()))
		{
			throw new RuntimeException(Text::_('COM_ATS_ATTACHMENTS_ERR_INVALIDPOST'));
		}

		$this->post = $post;

		return $this;
	}

	/**
	 * Runs after deleting an attachment record.
	 *
	 * Used to delete the attachment files themselves.
	 *
	 * @param   bool      $result  Did the record actually delete?
	 * @param   int|null  $pk      The ID of the attachment which was deleted.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onAfterDelete(bool &$result, ?int $pk): void
	{
		// Get the mangled filename stored in onBeforeDelete
		$fileName = self::$filesToDelete[$pk] ?? null;

		// No filename found? Nothing to do here!
		if (empty($fileName))
		{
			return;
		}

		// Remove the mangled filename from the stash.
		unset(self::$filesToDelete[$pk]);

		// If the record wasn't actualyl deleted I have nothing to do here.
		if (!$result)
		{
			return;
		}

		// Get the absolute filename to the file to delete. Returns NULL if this no longer exists.
		$filePath = $this->getAbsoluteFilename($fileName);

		// If it's NULL I have nothign to do here.
		if (empty($filePath))
		{
			return;
		}

		// Try (really hard) to delete the attachment file.
		try
		{
			$deleted = File::delete($filePath);
		}
		catch (\Exception $e)
		{
			$deleted = false;
		}

		if (!$deleted)
		{
			@unlink($filePath);
		}
	}

	/**
	 * Runs after resetting this object.
	 *
	 * Used to reset internal data.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onAfterReset(): void
	{
		$this->post = null;
	}

	/**
	 * Runs before deleting the attachment record.
	 *
	 * Used to stash the mangled filename of the attachment so I can actually delete it onAfterDelete.
	 *
	 * @param   int|null  $pk  The ID of the attachment being deleted.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeDelete(?int &$pk): void
	{
		$pk = $pk ?? $this->getId();

		$attachment = $this;

		if ($pk != $this->getId())
		{
			$attachment = clone $this;

			if (!$attachment->load($pk))
			{
				return;
			}
		}

		self::$filesToDelete[$pk] = $attachment->mangled_filename;
	}
}