<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Table;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\Filter;
use Akeeba\Component\ATS\Administrator\Mixin\TableAssertionTrait;
use Joomla\Database\DatabaseDriver;
use Joomla\Event\DispatcherInterface;

/**
 * Class AutoreplyTable
 *
 * @property    int    $id
 * @property    string $title
 * @property    string $reply
 * @property    int    $ordering
 * @property    int    $enabled
 * @property    int    $run_after_manager
 * @property    int    $num_posts
 * @property    int    $min_after
 * @property    int    $attachment
 * @property    string $keywords_title
 * @property    string $keywords_text
 * @property    array  $params
 *
 * @since   5.0.0
 */
class AutoreplyTable extends AbstractTable
{
	use TableAssertionTrait;

	/**
	 * Public constructor
	 *
	 * @param   DatabaseDriver            $db          Database driver object to Joomla's DB
	 * @param   DispatcherInterface|null  $dispatcher  Joomla event dispatcher
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	public function __construct(DatabaseDriver $db, ?DispatcherInterface $dispatcher  = null)
	{
		$this->_supportNullValue = true;

		parent::__construct('#__ats_autoreplies', 'id', $db, $dispatcher);

		$this->setColumnAlias('published', 'enabled');

		$this->params = [];
	}

	/**
	 * Runs right after resetting this table object.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onAfterReset(): void
	{
		$this->params = [];
	}

	/**
	 * Runs before saving the data to the database.
	 *
	 * @param   bool  $result       Did the save succeed?
	 * @param   bool  $updateNulls  Should I also update fields whose value is NULL?
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onAfterStore(bool &$result, bool &$updateNulls): void
	{
		$this->params = @json_decode($this->params ?? '{}', true) ?? [];
	}

	/**
	 * Runs before binding raw data to the object.
	 *
	 * @param   array|object  $src     Source data to bind
	 * @param   array         $ignore  Keys to ignore when binding
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeBind(&$src, array &$ignore = []): void
	{
		$src = (array) $src;

		$src['params']               = $src['params'] ?? [];

		if (is_string($src['params']))
		{
			$src['params'] = @json_decode($src['params'], true) ?? [];
		}

		$src['params']['categories'] = $src['params']['categories'] ?? [];
	}

	/**
	 * Checks and sanitizes data before saving them.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeCheck(): void
	{
		$this->title = trim($this->title ?? '');
		$this->reply = Filter::filterText($this->reply ?? '');

		$this->assertNotEmpty($this->title, 'COM_ATS_AUTOREPLIES_ERR_TITLE');
		$this->assertNotEmpty($this->reply, 'COM_ATS_AUTOREPLY_ERR_REPLY_EMPTY');
	}

	/**
	 * Runs before saving the data to the database.
	 *
	 * @param   bool  $updateNulls  Should I also update fields whose value is NULL?
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeStore(bool &$updateNulls): void
	{
		$this->params = @json_encode($this->params ?: []) ?? '{}';
	}
}