<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Table;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\Filter;
use Akeeba\Component\ATS\Administrator\Mixin\TableAssertionTrait;
use Akeeba\Component\ATS\Administrator\Mixin\TableCreateModifyTrait;
use InvalidArgumentException;
use Joomla\CMS\Language\Text;
use Joomla\Database\DatabaseDriver;
use Joomla\Event\DispatcherInterface;
use RuntimeException;

/**
 * Class ManagernoteTable
 *
 * @property int    $id            Manager note ID
 * @property int    $ticket_id     Ticket this manager note belongs to
 * @property string $note_html     Manager note content
 * @property string $created       Created date/time, UTC
 * @property int    $created_by    User ID who created this manager note
 * @property string $modified      Last modified date/time, UTC
 * @property int    $modified_by   User ID who last modified this manager note
 * @property int    $enabled       Is this manager note published?
 *
 * @since   5.0.0
 */
class ManagernoteTable extends AbstractTable
{
	use TableCreateModifyTrait
	{
		TableCreateModifyTrait::onBeforeStore as onBeforeStoreCreateModifyAware;
	}
	use TableAssertionTrait;

	/**
	 * Is this a new Manager Note?
	 *
	 * @var   bool
	 * @since 5.0.0
	 */
	private $isNewNote = true;

	/**
	 * Ticket this manager note belongs to
	 *
	 * @var   TicketTable|null
	 * @since 5.0.0
	 */
	private $ticket;

	/**
	 * Public constructor
	 *
	 * @param   DatabaseDriver            $db          Database driver object to Joomla's DB
	 * @param   DispatcherInterface|null  $dispatcher  Joomla event dispatcher
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	public function __construct(DatabaseDriver $db, ?DispatcherInterface $dispatcher  = null)
	{
		$this->_supportNullValue = true;

		$this->setColumnAlias('published', 'enabled');

		parent::__construct('#__ats_managernotes', 'id', $db, $dispatcher);
	}

	/**
	 * Get the ticket this manager note belongs to
	 *
	 * @return  TicketTable|null
	 * @since   5.0.0
	 */
	public function getTicket(): ?TicketTable
	{
		if (is_null($this->ticket))
		{
			$this->ticket = new TicketTable($this->getDbo(), $this->getDispatcher());

			if ($this->ticket->load($this->ticket_id) === false)
			{
				throw new RuntimeException(Text::_('COM_ATS_POSTS_ERR_NOTICKET'));
			}
		}

		return $this->ticket;
	}

	/**
	 * Set the ticket this manager note belongs to.
	 *
	 * @param   TicketTable|null  $ticket  The ticket to set. NULL to make ATS reload the ticket on the next getTicket()
	 *                                     method call.
	 * @param   bool              $force   True to reset $this->ticket_id to the $ticket ID (or NULL, if no ticket).
	 *
	 * @return  self
	 * @since   5.0.0
	 */
	public function setTicket(?TicketTable $ticket, bool $force = false): self
	{
		$this->ticket = $ticket;

		if ($force)
		{
			$this->ticket_id = empty($ticket) ? null : $ticket->getId();

			return $this;
		}

		if (empty($ticket) && $this->ticket->getId() != $this->ticket_id)
		{
			throw new InvalidArgumentException(Text::_('COM_ATS_POSTS_ERR_INVALIDTICKET'));
		}

		return $this;
	}

	/**
	 * Checks and sanitizes data before saving them.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeCheck(): void
	{
		$this->assertNotEmpty($this->ticket_id, Text::_('COM_ATS_POSTS_ERR_NOTICKET'));

		$ticket = $this->getTicket();

		$this->assert($ticket->getId() == $this->ticket_id, Text::_('COM_ATS_POSTS_ERR_INVALIDTICKET'));

		$this->note_html = Filter::filterText($this->note_html);
	}

	/**
	 * Runs right before resetting this table object. Resets internal data.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeReset(): void
	{
		$this->ticket    = null;
		$this->isNewNote = true;
	}

	/**
	 * Runs before saving the data to the database.
	 *
	 * Sets the created and modified fields.
	 *
	 * Sets a flag indicating if this is a new or existing manager note.
	 *
	 * @param   bool  $updateNulls  Should I also update fields whose value is NULL?
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeStore(bool &$updateNulls): void
	{
		$this->onBeforeStoreCreateModifyAware();

		// Save the ticket ID before the data is saved.
		$this->isNewNote = empty($this->id);
	}
}