<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Table;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\Filter;
use Akeeba\Component\ATS\Administrator\Helper\Permissions;
use Akeeba\Component\ATS\Administrator\Mixin\TableAssertionTrait;
use Akeeba\Component\ATS\Administrator\Mixin\TableCreateModifyTrait;
use InvalidArgumentException;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\Database\DatabaseDriver;
use Joomla\Database\ParameterType;
use Joomla\Event\DispatcherInterface;
use Joomla\Registry\Registry;
use RuntimeException;

/**
 * Class PostTable
 *
 * @property int    $id            Post ID
 * @property string $attachment_id List of attachment IDs
 * @property int    $ticket_id     Ticket this post belongs to
 * @property string $content_html  Post content
 * @property string $origin        Origin of this post: web, email
 * @property float  $timespent     How much time was spent on this post (only on staff replies)
 * @property string $email_uid     Email message UID (only for origin=email)
 * @property string $created       Created date/time, UTC
 * @property int    $created_by    User ID who created this post
 * @property string $modified      Last modified date/time, UTC
 * @property int    $modified_by   User ID who last modified this post
 * @property int    $enabled       Is this post published?
 *
 * @since   5.0.0
 */
class PostTable extends AbstractTable
{
	use TableCreateModifyTrait
	{
		TableCreateModifyTrait::onBeforeStore as onBeforeStoreCreateModifyAware;
	}
	use TableAssertionTrait;

	/**
	 * Attachments for this post
	 *
	 * @var   AttachmentTable[]
	 * @since 5.0.0
	 */
	private $attachments = [];

	/**
	 * Should I check if the ticket already exists when saving this post?
	 *
	 * @var   bool
	 * @since 5.0.0
	 */
	private $checkTicketExists = true;

	/**
	 * Is this a new post?
	 *
	 * @var   bool
	 * @since 5.0.0
	 */
	private $isNewPost = true;

	/**
	 * Ticket this post belongs to
	 *
	 * @var   TicketTable|null
	 * @since 5.0.0
	 */
	private $ticket;

	/**
	 * Public constructor
	 *
	 * @param   DatabaseDriver            $db          Database driver object to Joomla's DB
	 * @param   DispatcherInterface|null  $dispatcher  Joomla event dispatcher
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	public function __construct(DatabaseDriver $db, ?DispatcherInterface $dispatcher  = null)
	{
		$this->_supportNullValue = true;

		parent::__construct('#__ats_posts', 'id', $db, $dispatcher);

		$this->setColumnAlias('published', 'enabled');

		$this->params = new Registry();
	}

	/**
	 * Get the attachments to this post.
	 *
	 * @return AttachmentTable[]
	 *
	 * @since  5.0.0
	 */
	public function getAttachments(): array
	{
		if (!defined('ATS_PRO') || !ATS_PRO)
		{
			return [];
		}

		if (empty($this->attachments))
		{
			$this->loadAttachments();
		}

		return $this->attachments;
	}

	/**
	 * Set attachments to this ticket.
	 *
	 * When $replaceExisting is false only attachments whose ID matches on of the IDs in the $this->attachment_id comma–
	 * separated list of attachment IDs will be set into the $this->attachments property.
	 *
	 * When $replaceExisting is true the entire $attachments array will replace $this->attachments AND their IDs will be
	 * used to re–populate the $this->attachment_id property.
	 *
	 * @param   AttachmentTable[]  $attachments      The attachments to assign to this post
	 * @param   bool               $replaceExisting  Forcibly replace all attachments and update attachment_id?
	 *
	 * @return  self
	 * @since   5.0.0
	 */
	public function setAttachments(array $attachments, bool $replaceExisting = false): self
	{
		if (!defined('ATS_PRO') || !ATS_PRO)
		{
			return $this;
		}

		if ($replaceExisting)
		{
			$this->attachments   = $attachments;
			$this->attachment_id = implode(',', array_map(function (AttachmentTable $attachment) {
				return $attachment->getId();
			}, $attachments));

			return $this;
		}

		$this->attachments = [];

		if (empty($this->attachment_id))
		{
			return $this;
		}

		$ids               = explode(',', $this->attachment_id);
		$this->attachments = array_filter($attachments, function (AttachmentTable $attachment) use ($ids) {
			return in_array($attachment->getId(), $ids);
		});

		return $this;
	}

	/**
	 * Get the ticket this post belongs to
	 *
	 * @return  TicketTable|null
	 * @since   5.0.0
	 */
	public function getTicket(): ?TicketTable
	{
		if (is_null($this->ticket))
		{
			$this->ticket = new TicketTable($this->getDbo(), $this->getDispatcher());

			if ($this->ticket->load($this->ticket_id) === false)
			{
				throw new RuntimeException(Text::_('COM_ATS_POSTS_ERR_NOTICKET'));
			}
		}

		return $this->ticket;
	}

	/**
	 * Set the ticket this post belongs to.
	 *
	 * @param   TicketTable|null  $ticket  The ticket to set. NULL to make ATS reload the ticket on the next getTicket()
	 *                                     method call.
	 * @param   bool              $force   True to reset $this->ticket_id to the $ticket ID (or NULL, if no ticket).
	 *
	 * @return  self
	 * @since   5.0.0
	 */
	public function setTicket(?TicketTable $ticket, bool $force = false): self
	{
		$this->ticket = $ticket;

		if ($force)
		{
			$this->ticket_id = empty($ticket) ? null : $ticket->getId();

			return $this;
		}

		if (empty($ticket) && $this->ticket->getId() != $this->ticket_id)
		{
			throw new InvalidArgumentException(Text::_('COM_ATS_POSTS_ERR_INVALIDTICKET'));
		}

		return $this;
	}

	/**
	 * Runs after deleting a post.
	 *
	 * Used to also delete attachments.
	 *
	 * @param   bool      $result  Did the post actually delete?
	 * @param   int|null  $pk      The post ID which was deleted.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onAfterDelete(bool &$result, ?int $pk): void
	{
		if (!$result)
		{
			return;
		}

		$pk = $pk ?? $this->getId();

		if (defined('ATS_PRO') && ATS_PRO)
		{
			// Get all attachments for the post which was just deleted
			$db    = $this->getDbo();
			$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
				->select('*')
				->from($db->quoteName('#__ats_attachments'))
				->where($db->quoteName('post_id') . ' = :postId')
				->bind(':postId', $pk, ParameterType::INTEGER);

			// Delete any attachments found
			foreach ($db->setQuery($query)->loadAssocList() ?: [] as $attachmentData)
			{
				$attachment = new AttachmentTable($this->getDbo(), $this->getDispatcher());
				$attachment->bind($attachmentData);
				$attachment->delete();
			}
		}
	}

	/**
	 * Runs after saving a post to the database
	 *
	 * @param   bool  $result       Did the post actually save?
	 * @param   bool  $updateNulls  Was I asked to update fields whose values were set to NULL?
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onAfterStore(bool &$result, bool $updateNulls): void
	{
		if (!$result)
		{
			return;
		}

		// Update the ticket when we submit a new post to a ticket that's NOT closed.
		$ticket = $this->getTicket();

		if ($this->isNewPost && $ticket->status != 'C')
		{
			// Set the ticket status depending on who made this post. User: Open. Manager: Pending.
			$ticketChanges = [
				'status'      => ($this->created_by == $ticket->created_by) ? 'O' : 'P',
				'modified'    => (clone Factory::getDate())->toSql(),
				'modified_by' => $this->created_by,
			];

			// Get the total amount of tracked item for all posts of this ticket
			$db = $this->getDbo();

			$query     = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
				->select('SUM(' . $db->quoteName('timespent') . ')')
				->from($db->quoteName('#__ats_posts'))
				->where($db->quoteName('ticket_id') . ' = :ticketId')
				->where($db->quoteName('enabled') . ' = ' . $db->quote('1'))
				->bind(':ticketId', $this->ticket_id, ParameterType::INTEGER);
			$ticketChanges['timespent'] = $db->setQuery($query)->loadResult() ?: 0.0;

			// If the ticket is unassigned and the poster is a manager in the category auto–assign the ticket to them.
			if (empty($ticket->assigned_to) && Permissions::isManager($ticket->catid, $this->created_by))
			{
				$ticketChanges['assigned_to'] = $this->created_by;
			}

			TicketTable::$noPermissionsCheck = true;

			$ticket->save($ticketChanges);

			TicketTable::$noPermissionsCheck = false;
		}
	}

	/**
	 * Runs before binding raw data to the object.
	 *
	 * @param   array|object  $src     Source data to bind
	 * @param   array|string  $ignore  Keys to ignore when binding
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeBind(&$src, $ignore = ''): void
	{
		$src = (array) $src;

		// handle the _do_not_check_ticket flag in the incoming data
		if (isset($src['_do_not_check_ticket']))
		{
			$this->checkTicketExists = false;

			unset($src['_do_not_check_ticket']);
		}
	}

	/**
	 * Checks and sanitizes data before saving them.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeCheck(): void
	{
		if ($this->checkTicketExists)
		{
			$this->assertNotEmpty($this->ticket_id, Text::_('COM_ATS_POSTS_ERR_NOTICKET'));

			$ticket = $this->getTicket();

			$this->assert($ticket->getId() == $this->ticket_id, Text::_('COM_ATS_POSTS_ERR_INVALIDTICKET'));
		}

		$this->content_html = Filter::filterText($this->content_html);

		$this->timespent = floatval($this->timespent) ?: 0.0;
	}

	/**
	 * Runs right before resetting this table object. Resets internal data.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeReset(): void
	{
		$this->attachments       = [];
		$this->ticket            = null;
		$this->isNewPost         = true;
		$this->checkTicketExists = true;
	}

	/**
	 * Runs before saving the data to the database.
	 *
	 * Sets the created and modified fields.
	 *
	 * Sets a flag indicating if this is a new or existing post.
	 *
	 * @param   bool  $updateNulls  Should I also update fields whose value is NULL?
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeStore(bool &$updateNulls): void
	{
		$this->onBeforeStoreCreateModifyAware();

		// Save the ticket ID before the data is saved.
		$this->isNewPost = empty($this->id);
	}

	/**
	 * Loads the attachments into $this->attachments.
	 *
	 * @return void
	 * @internal
	 *
	 * @since  5.0.0
	 */
	private function loadAttachments(): void
	{
		$this->attachments = [];

		if (!defined('ATS_PRO') || !ATS_PRO)
		{
			return;
		}

		if (empty($this->attachment_id))
		{
			return;
		}

		$ids = explode(',', $this->attachment_id);

		foreach ($ids as $id)
		{
			$attachment = new AttachmentTable($this->getDbo(), $this->getDispatcher());

			if (!$attachment->load($id))
			{
				continue;
			}

			$this->attachments[] = $attachment;
		}
	}
}