<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\Table;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\Filter;
use Akeeba\Component\ATS\Administrator\Helper\Permissions;
use Akeeba\Component\ATS\Administrator\Mixin\CMSObjectWorkaroundTrait;
use Akeeba\Component\ATS\Administrator\Mixin\TableAssertionTrait;
use Akeeba\Component\ATS\Administrator\Mixin\TableCreateModifyTrait;
use Akeeba\Component\ATS\Administrator\Model\EmailSending;
use Akeeba\Component\ATS\Site\Service\Category;
use Exception;
use Joomla\CMS\Event\AbstractEvent;
use Joomla\CMS\Factory;
use Joomla\CMS\Helper\CMSHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Table\CoreContent;
use Joomla\CMS\Tag\TaggableTableInterface;
use Joomla\CMS\Tag\TaggableTableTrait;
use Joomla\CMS\UCM\UCMContent;
use Joomla\Database\DatabaseDriver;
use Joomla\Database\DatabaseInterface;
use Joomla\Database\ParameterType;
use Joomla\Event\DispatcherInterface;
use Joomla\Registry\Registry;
use RuntimeException;

/**
 * Class TicketTable
 *
 * @property  int|null $id           Primary ID
 * @property  int      $catid        Category ID
 * @property  string   $status       Ticket status: O, P, C, 1...99
 * @property  string   $title        Human–readable ticket title
 * @property  string   $alias        Ticket alias
 * @property  int      $public       Public (1) or Private (0)
 * @property  int      $priority     Ticket priority
 * @property  string   $origin       Ticket origin: web, email, ...
 * @property  int      $assigned_to  User ID the ticket is assigned to
 * @property  float    $timespent    Time spent on the ticket, in minutes
 * @property  string   $created      Created date and time
 * @property  int      $created_by   Created by user ID
 * @property  string   $modified     Last modified date and time
 * @property  int      $modified_by  Modified by user ID
 * @property  int      $enabled      Displayed in the frontend (1) or not (0)
 * @property  Registry $params       Various other parameters
 *
 * @since   5.0.0
 */
class TicketTable extends AbstractTable implements TaggableTableInterface
{
	use TaggableTableTrait;
	use TableCreateModifyTrait
	{
		TableCreateModifyTrait::onBeforeStore as onBeforeStoreCreateModifyAware;
	}
	use TableAssertionTrait;
	use CMSObjectWorkaroundTrait;

	/**
	 * When true ATS will completely ignore user permissions when saving a ticket.
	 *
	 * This has precedence over the $overrideuser flag above.
	 *
	 * @var   bool
	 * @since 5.0.0
	 */
	public static $noPermissionsCheck = false;

	/**
	 * When false: ATS checks the permissions of the currently logged in user when saving.
	 *
	 * When true: ATS checks the permissions of the user specified in the created_by field  when saving.
	 *
	 * @var   bool
	 * @since 5.0.0
	 */
	public static $overrideuser = false;

	/**
	 * Cache of category IDs to category names
	 *
	 * @var   array
	 * @since 5.0.0
	 */
	private static $categoryNames = [];

	/**
	 * The UCM type alias. Used for tags, content versioning etc. Leave blank to effectively disable these features.
	 *
	 * @var    string
	 * @since  4.0.0
	 */
	public $typeAlias = 'com_ats.ticket';

	protected $newTags;

	/**
	 * Is this a new ticket?
	 *
	 * @var   bool
	 * @since 5.0.0
	 */
	private $isNew = true;

	/**
	 * The table values after load. Used onAfterStore to determine what's changed and trigger events.
	 *
	 * @var   array|null
	 * @since 3.0.0
	 */
	private $valuesOnLoad;

	/**
	 * Public constructor
	 *
	 * @param   DatabaseDriver            $db          Database driver object to Joomla's DB
	 * @param   DispatcherInterface|null  $dispatcher  Joomla event dispatcher
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	public function __construct(DatabaseDriver $db, ?DispatcherInterface $dispatcher  = null)
	{
		$this->_supportNullValue = true;

		parent::__construct('#__ats_tickets', 'id', $db, $dispatcher);

		$this->setColumnAlias('published', 'enabled');
		$this->setColumnAlias('fieldscatid', 'catid');

		$this->params = new Registry();
	}

	/**
	 * Method to change the status for a row or list of rows in the database table.
	 *
	 * @param   mixed    $pks     An optional array of primary key values to update. If not set the instance property
	 *                            value is used.
	 * @param   string   $status  The status of the ticket: O, P, C, 1...99
	 * @param   integer  $userId  The user ID of the user performing the operation.
	 *
	 * @return  boolean  True on success; false if $pks is empty.
	 *
	 * @since   5.0.0
	 */
	public function changeStatus($pks = null, string $status = 'C', int $userId = 0): bool
	{
		// Sanitize input
		$userId = (int) $userId;

		// Pre-processing by observers
		$event = AbstractEvent::create(
			'onTableBeforeStatusChange',
			[
				'subject' => $this,
				'pks'     => $pks,
				'state'   => $status,
				'userId'  => $userId,
			]
		);
		$this->getDispatcher()->dispatch('onTableBeforeStatusChange', $event);

		if (!\is_null($pks))
		{
			if (!\is_array($pks))
			{
				$pks = [$pks];
			}

			foreach ($pks as $key => $pk)
			{
				if (!\is_array($pk))
				{
					$pks[$key] = [$this->_tbl_key => $pk];
				}
			}
		}

		// If there are no primary keys set check to see if the instance key is set.
		if (empty($pks))
		{
			$pk = [];

			foreach ($this->_tbl_keys as $key)
			{
				if ($this->$key)
				{
					$pk[$key] = $this->$key;
				}
				// We don't have a full primary key - return false
				else
				{
					$this->setErrorOrThrow(Text::_('JLIB_DATABASE_ERROR_NO_ROWS_SELECTED'));

					return false;
				}
			}

			$pks = [$pk];
		}

		$statusField = $this->getColumnAlias('status');

		foreach ($pks as $pk)
		{
			// Update the public state for rows with the given primary keys.
			$db    = method_exists($this, 'getDbo') ? $this->getDbo() : $this->_db;
			$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
				->update($this->_tbl)
				->set($db->quoteName($statusField) . ' = :status')
				->bind(':status', $status, ParameterType::STRING);

			// Build the WHERE clause for the primary keys.
			$this->appendPrimaryKeys($query, $pk);

			$db->setQuery($query);

			try
			{
				$db->execute();
			}
			catch (\RuntimeException $e)
			{
				$this->setErrorOrThrow($e->getMessage());

				return false;
			}

			// If the Table instance value is in the list of primary keys that were set, set the instance.
			$ours = true;

			foreach ($this->_tbl_keys as $key)
			{
				if ($this->$key != $pk[$key])
				{
					$ours = false;
				}
			}

			if ($ours)
			{
				$this->$statusField = $status;
			}
		}

		// Pre-processing by observers
		$event = AbstractEvent::create(
			'onTableAfterStatusChange',
			[
				'subject' => $this,
				'pks'     => $pks,
				'state'   => $status,
				'userId'  => $userId,
			]
		);
		$this->getDispatcher()->dispatch('onTableAfterStatusChange', $event);

		return true;
	}

	/**
	 * Returns the name of the category the ticket belongs to (assuming a positive integer catid property)
	 *
	 * @return  string|null  The category name. NULL if it doesn't exist, it's not an ATS category or catid is invalid.
	 *
	 * @since   5.0.0
	 */
	public function getCategoryName(): ?string
	{
		if (empty($this->catid) || $this->catid <= 0)
		{
			return null;
		}

		if (isset(self::$categoryNames[$this->catid]))
		{
			return self::$categoryNames[$this->catid];
		}

		$cat = (new Category([]))->get($this->catid);

		self::$categoryNames[$this->catid] = $cat ? $cat->title : null;

		return self::$categoryNames[$this->catid];
	}

	/**
	 * Get the type alias for the tags mapping table
	 *
	 * The type alias generally is the internal component name with the
	 * content type. Ex.: com_content.article
	 *
	 * @return  string  The alias as described above
	 *
	 * @since   5.0.0
	 */
	public function getTypeAlias()
	{
		return 'com_ats.ticket';
	}

	/**
	 * Is this a new ticket?
	 *
	 * @return  bool
	 * @since   5.0.0
	 */
	public function isNew(): bool
	{
		return $this->isNew;
	}

	/**
	 * Method to set the public state for a row or list of rows in the database table.
	 *
	 * @param   mixed    $pks     An optional array of primary key values to update. If not set the instance property
	 *                            value is used.
	 * @param   integer  $state   The public state. eg. [0 = private, 1 = public]
	 * @param   integer  $userId  The user ID of the user performing the operation.
	 *
	 * @return  boolean  True on success; false if $pks is empty.
	 *
	 * @since   5.0.0
	 */
	public function makepublic($pks = null, $state = 1, $userId = 0): bool
	{
		// Sanitize input
		$userId = (int) $userId;
		$state  = (int) $state;

		// Pre-processing by observers
		$event = AbstractEvent::create(
			'onTableBeforePublicChange',
			[
				'subject' => $this,
				'pks'     => $pks,
				'state'   => $state,
				'userId'  => $userId,
			]
		);
		$this->getDispatcher()->dispatch('onTableBeforePublicChange', $event);

		if (!\is_null($pks))
		{
			if (!\is_array($pks))
			{
				$pks = [$pks];
			}

			foreach ($pks as $key => $pk)
			{
				if (!\is_array($pk))
				{
					$pks[$key] = [$this->_tbl_key => $pk];
				}
			}
		}

		// If there are no primary keys set check to see if the instance key is set.
		if (empty($pks))
		{
			$pk = [];

			foreach ($this->_tbl_keys as $key)
			{
				if ($this->$key)
				{
					$pk[$key] = $this->$key;
				}
				// We don't have a full primary key - return false
				else
				{
					$this->setErrorOrThrow(Text::_('JLIB_DATABASE_ERROR_NO_ROWS_SELECTED'));

					return false;
				}
			}

			$pks = [$pk];
		}

		$publicField = $this->getColumnAlias('public');

		foreach ($pks as $pk)
		{
			// Update the public state for rows with the given primary keys.
			$db    = method_exists($this, 'getDbo') ? $this->getDbo() : $this->_db;
			$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
				->update($this->_tbl)
				->set($db->quoteName($publicField) . ' = ' . (int) $state);

			// Build the WHERE clause for the primary keys.
			$this->appendPrimaryKeys($query, $pk);

			$db->setQuery($query);

			try
			{
				$db->execute();
			}
			catch (\RuntimeException $e)
			{
				$this->setErrorOrThrow($e->getMessage());

				return false;
			}

			// If the Table instance value is in the list of primary keys that were set, set the instance.
			$ours = true;

			foreach ($this->_tbl_keys as $key)
			{
				if ($this->$key != $pk[$key])
				{
					$ours = false;
				}
			}

			if ($ours)
			{
				$this->$publicField = $state;
			}
		}

		// Pre-processing by observers
		$event = AbstractEvent::create(
			'onTableAfterPublicChange',
			[
				'subject' => $this,
				'pks'     => $pks,
				'state'   => $state,
				'userId'  => $userId,
			]
		);
		$this->getDispatcher()->dispatch('onTableAfterPublicChange', $event);

		return true;
	}

	/**
	 * Get the Manager Notes belonging to a ticket, ordered by ID ascending
	 *
	 * @param   int|null  $ticketId  The ticket to load notes for. NULL for current ticket object's ID.
	 *
	 * @return  ManagernoteTable[]
	 *
	 * @since   5.0.0
	 */
	public function managerNotes(?int $ticketId = null): array
	{
		if (!defined('ATS_PRO') || !ATS_PRO)
		{
			return [];
		}

		// Make sure we have a ticket ID
		$ticketId = $ticketId ?? $this->id;

		// No ticket ID and this object is not a concrete ticket: bail out.
		if (empty($ticketId))
		{
			return [];
		}

		// Get the ticket object
		$ticket = ($ticketId == $this->id) ? $this : null;

		if (empty($ticket))
		{
			$ticket = clone $this;

			if (!$ticket->load($ticketId))
			{
				return [];
			}
		}

		// Get all manager notes of a ticket
		$db    = $this->getDbo();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('*')
			->from($db->quoteName('#__ats_managernotes'))
			->where($db->quoteName('ticket_id') . ' = :ticketId')
			->order($db->quoteName('id') . ' ASC')
			->bind(':ticketId', $ticketId, ParameterType::INTEGER);

		$allNotesData = $db->setQuery($query)->loadAssocList() ?: [];

		// Convert raw notes data to notes objects
		$noteTable = new ManagernoteTable($this->getDbo(), $this->getDispatcher());

		return array_map(function (array $postData) use ($noteTable, $ticket) {
			$note = clone $noteTable;
			$note->reset();
			$note->bind($postData);
			$note->setTicket($ticket);

			return $note;
		}, $allNotesData);
	}

	/**
	 * Bind data with the same semantics as load() would.
	 *
	 * Basically, it sets the valuesOnLoad internal table (if it's not already set) so that mail events work correctly.
	 *
	 * @param   array|object  $source
	 *
	 * @return  void
	 * @since   5.3.10
	 */
	public function bindAsLoadEquivalent($source): void
	{
		if (empty($this->valuesOnLoad))
		{
			$this->valuesOnLoad = $this->filterByTableFields((array) $source);
		}

		$this->bind($source);
	}

	/**
	 * Get the posts belonging to a ticket, ordered by ID ascending
	 *
	 * @param   int|null  $ticketId         The ticket to load posts for. NULL for current ticket object's ID.
	 * @param   bool      $withAttachments  Should I eager load attachments as well?
	 *
	 * @return  PostTable[]
	 *
	 * @since   5.0.0
	 */
	public function posts(?int $ticketId, bool $withAttachments = true): array
	{
		// Make sure we have a ticket ID
		$ticketId = $ticketId ?? $this->id;

		// No ticket ID and this object is not a concrete ticket: bail out.
		if (empty($ticketId))
		{
			return [];
		}

		// Get the ticket object
		$ticket = ($ticketId == $this->id) ? $this : null;

		if (empty($ticket))
		{
			$ticket = clone $this;

			if (!$ticket->load($ticketId))
			{
				return [];
			}
		}

		// Get all posts of a ticket
		$db    = $this->getDbo();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('*')
			->from($db->quoteName('#__ats_posts'))
			->where($db->quoteName('ticket_id') . ' = :ticketId')
			->order($db->quoteName('id') . ' ASC')
			->bind(':ticketId', $ticketId, ParameterType::INTEGER);

		$allPostData = $db->setQuery($query)->loadAssocList() ?: [];

		// Get all attachments, if necessary
		$attachments = [];

		if (defined('ATS_PRO') && ATS_PRO && $withAttachments && !empty($allPostData))
		{
			$postIDs = array_map(function (array $postData) {
				return $postData['id'];
			}, $allPostData);

			$query           = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
				->select('*')
				->from($db->quoteName('#__ats_attachments'))
				->whereIn($db->quoteName('post_id'), $postIDs)
				->order($db->quoteName('id') . ' ASC');
			$attachmentTable = new AttachmentTable($this->getDbo(), $this->getDispatcher());
			$attachments     = array_map(function (array $attachmentData) use ($attachmentTable) {
				$attachment = clone $attachmentTable;
				$attachment->reset();
				$attachment->bind($attachmentData);

				return $attachment;
			}, $db->setQuery($query)->loadAssocList() ?: []);
		}

		// Convert raw post data to post objects
		$postTable = new PostTable($this->getDbo(), $this->getDispatcher());

		return array_map(function (array $postData) use ($postTable, $ticket, $attachments) {
			$post = clone $postTable;
			$post->reset();
			$post->bind($postData);
			$post->setTicket($ticket);

			if (defined('ATS_PRO') && ATS_PRO)
			{
				$post->setAttachments($attachments);
			}

			return $post;
		}, $allPostData);
	}

	/**
	 * Runs after binding data to the object
	 *
	 * @param   bool          $result  Did the bind succeed?
	 * @param   array|object  $src     Source data
	 * @param   array|string  $ignore  Fields we were asked to ignore. Array or space separated string.
	 *
	 * @return  void
	 * @since   5.0.0
	 */
	protected function onAfterBind(bool &$result, $src, $ignore): void
	{
		$this->isNew = $result && empty($this->id);

		if ($this->isNew)
		{
			$this->id = null;
		}
	}

	/**
	 * Runs after deleting a ticket.
	 *
	 * Used to also delete posts and manager notes.
	 *
	 * @param   bool      $result  Did the post actually delete?
	 * @param   int|null  $pk      The ticket ID which was deleted.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onAfterDelete(bool &$result, ?int $pk): void
	{
		if (!$result)
		{
			return;
		}

		$pk = $pk ?? $this->id;

		// Get all posts for the ticket which was just deleted
		$db    = $this->getDbo();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('*')
			->from($db->quoteName('#__ats_posts'))
			->where($db->quoteName('ticket_id') . ' = :ticketId')
			->bind(':ticketId', $pk, ParameterType::INTEGER);

		// Delete any posts found under that ticket
		foreach ($db->setQuery($query)->loadAssocList() ?: [] as $postData)
		{
			$post = new PostTable($this->getDbo(), $this->getDispatcher());
			$post->bind($postData);
			$post->delete();
		}

		// Get all manager notes for the ticket which was just deleted
		$db    = $this->getDbo();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('*')
			->from($db->quoteName('#__ats_managernotes'))
			->where($db->quoteName('ticket_id') . ' = :ticketId')
			->bind(':ticketId', $pk, ParameterType::INTEGER);

		// Delete any posts found under that ticket
		foreach ($db->setQuery($query)->loadAssocList() ?: [] as $noteData)
		{
			$managerNote = new ManagernoteTable($this->getDbo(), $this->getDispatcher());
			$managerNote->bind($noteData);
			$managerNote->delete();
		}
	}

	/**
	 * Runs after loading a record from the database
	 *
	 * @param   bool   $result  Did the record load?
	 * @param   mixed  $keys    The keys used to load the record.
	 * @param   bool   $reset   Was I asked to reset the object before loading the record?
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onAfterLoad(bool &$result, $keys, bool $reset): void
	{
		$this->isNew = $result && empty($this->id);

		if (!$this->isNew)
		{
			$this->ensureUcmRecord();
		}

		$this->valuesOnLoad = $this->filterByTableFields($this->getProperties());
	}

	/**
	 * Runs after the table is reset with reset()
	 *
	 * Used to reset the params Registry object.
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onAfterReset(): void
	{
		$this->isNew        = true;
		$this->params       = new Registry();
		$this->valuesOnLoad = [];
	}

	/**
	 * Runs after saving the ticket to the database.
	 *
	 * Used to convert the previously cast to JSON string params back to a registry  object.
	 *
	 * @param   bool  $result       Was saving successful?
	 * @param   bool  $updateNulls  Was I told to save NULL values?
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onAfterStore(bool &$result, bool $updateNulls): void
	{
		if (!is_object($this->params) || !($this->params instanceof Registry))
		{
			$this->params = new Registry($this->params ?? '{}');
		}

		$postSaveValues = $this->filterByTableFields($this->getProperties());
		$modifiedValues = array_diff_assoc($postSaveValues ?: [], $this->valuesOnLoad ?: []);

		if (!empty($modifiedValues))
		{
			$this->triggerEvent('onChangedValues', [$modifiedValues ?: [], $this->valuesOnLoad ?: []]);
		}

		$this->valuesOnLoad = $postSaveValues;
	}

	/**
	 * Runs before binding raw data to the object.
	 *
	 * Used to ensure params is a Registry object. If not, it's converted from JSON encoded data.
	 *
	 * @param   array|object  $src     Source data to bind
	 * @param   array|string  $ignore  Keys to ignore when binding
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeBind(&$src, &$ignore = ''): void
	{
		$src = (array) $src;

		if (!is_object($src['params'] ?? '') || !($src['params'] instanceof Registry))
		{
			$this->params = new Registry($src['params'] ?? '{}');
		}
	}

	/**
	 * Check and normalise the data before saving into the database
	 *
	 * @return  void
	 * @throws  RuntimeException  On validation error
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeCheck()
	{
		$app     = Factory::getApplication();
		$session = $app->getSession();

		$this->id          = $this->id ?: null;
		$this->assigned_to = $this->assigned_to ?: 0;
		$this->assertNotEmpty($this->catid, Text::_('COM_ATS_TICKETS_ERR_NOCATID'));

		// Make sure a non–empty assigned user is actually a manager of the category the ticket belongs to.
		if (!empty($this->assigned_to))
		{
			$this->assert(Permissions::isManager($this->catid, $this->assigned_to), 'COM_ATS_TICKETS_ERR_INVALID_ASSIGNED_TO');
		}

		if (!self::$noPermissionsCheck)
		{
			$userid = (self::$overrideuser || $session->get('com_ats.ticket.overrideuser', 0))
				? $this->created_by
				: null;

			$aclPrivileges = Permissions::getAclPrivileges($this->catid, $userid);

			$this->assert($aclPrivileges['core.create'], Text::_('COM_ATS_TICKETS_ERR_CATNOAUTH'));
		}

		$this->title = trim($this->title);
		$this->assertNotEmpty($this->title, Text::_('COM_ATS_TICKETS_ERR_NOTITLE'));

		$this->alias = is_string($this->alias) ? trim($this->alias) : '';
		$this->alias = $this->alias ?: Filter::toSlug($this->title);

		// If we have the same slug add numeric suffixes
		$aliasParts = explode('-', $this->alias);
		$lastPart   = array_pop($aliasParts);

		if (!is_numeric($lastPart))
		{
			$aliasParts[] = $lastPart;
		}

		$baseAlias  = implode('-', $aliasParts);
		$otherSlugs = $this->getSimilarSlugs($baseAlias, $this->id);

		if (in_array($this->alias, $otherSlugs))
		{
			$newAlias = $baseAlias;
			$counter  = 1;

			while (in_array($newAlias, $otherSlugs))
			{
				$newAlias = $baseAlias . '-' . $counter;
				$counter++;
			}

			$this->alias = $newAlias;
		}

		// Check the public status
		if (!$this->public)
		{
			$checkExisting = (clone $this);
			$checkExisting->reset();
			$allowed = !$checkExisting->load($this->id);
			$allowed = $allowed && ($checkExisting->getId() == $this->id);

			// Is the current user allowed to make the ticket private?
			if (!$allowed)
			{
				$action  = Permissions::getAclPrivileges($this->catid);
				$allowed = $action['ats.private'];
			}

			// Is the owner allowed private tickets?
			if (!$allowed)
			{
				$user    = Permissions::getUser($this->created_by);
				$allowed = $user->authorise('ats.private', 'com_ats.category.' . (int) ($this->catid));
			}

			// Switch to public mode if all checks failed
			if (!$allowed)
			{
				$this->public = 1;
			}
		}

		// Check the status
		$validStatuses = ['O', 'C', 'P'];

		for ($i = 1; $i <= 99; $i++)
		{
			$validStatuses[] = $i;
		}

		if (!in_array($this->status, $validStatuses))
		{
			$this->status = 'O';
		}

		// Check the origin
		if (!in_array($this->origin, ['web', 'email']))
		{
			$this->origin = 'web';
		}

		// Priority is not set, so I automatically set it by looking at the visibility
		if (!$this->priority)
		{
			if (!$this->public)
			{
				// Private tickets have high priority
				$this->priority = 1;
			}
			else
			{
				$this->priority = 5;
			}
		}

		$this->timespent = $this->timespent ?: 0.0;

		if (!is_numeric($this->timespent))
		{
			$this->timespent = floatval($this->timespent);
		}
	}

	/**
	 * Runs before saving the ticket to the database.
	 *
	 * Used to cast the params registry into a JSON string.
	 *
	 * @param   bool  $updateNulls  Should I update NULL values?
	 *
	 * @return  void
	 *
	 * @since   5.0.0
	 */
	protected function onBeforeStore(bool &$updateNulls): void
	{
		$this->isNew = empty($this->id);

		/**
		 * If this is an existing ticket we will NOT update the modified / modified_on fields. These fields are used to
		 * record the last person who replied to a ticket.
		 */
		if ($this->isNew)
		{
			$this->onBeforeStoreCreateModifyAware($updateNulls);
		}

		if (is_object($this->params) && ($this->params instanceof Registry))
		{
			$this->params = $this->params->toString('JSON');
		}
		elseif (is_array($this->params) || is_object($this->params))
		{
			$this->params = json_encode((array) $this->params);
		}
	}

	/**
	 * Internal event to handled changed values on save
	 *
	 * @param   array  $modifiedValues  A subset of the values being saved, containing only what's changed
	 * @param   array  $originalValues  The original values which were loaded from the database
	 *
	 * @since   5.0.0
	 */
	protected function onChangedValues(array $modifiedValues, array $originalValues)
	{
		// Send the emails when a ticket is assigned to a manager (but not ourselves; no need to notify us of our own action!)
		if (isset($modifiedValues['assigned_to']) && !empty($modifiedValues['assigned_to']) && ($modifiedValues['assigned_to'] != Permissions::getUser()->id))
		{
			(new EmailSending())->sendAssignedEmails($this);
		}
	}

	/**
	 * Makes sure that tickets created with older versions of Akeeba Ticket System have UCM records.
	 *
	 * UCM records are used for ticket tagging. You cannot tag an existing ticket unless it has a UCM record. For new
	 * tickets a UCM record is created automatically. For old, existing tickets we need to create a UCM record here.
	 *
	 * @since  5.0.0
	 */
	private function ensureUcmRecord(): void
	{
		$ucm = new UCMContent($this, $this->typeAlias);

		$genericHelper = new CMSHelper();
		$data          = $genericHelper->getRowData($this);
		$ucmData       = $ucm->mapData($data);

		$primaryId = $ucm->getPrimaryKey($ucmData['common']['core_type_id'], $ucmData['common']['core_content_item_id']);

		if (!empty($primaryId))
		{
			return;
		}

        $ucmContentTable = new CoreContent($this->getDbo());
		$ucmContentTable->save($ucmData['common']);
	}

	/**
	 * Get existing slugs which are identical to or start with the given slug
	 *
	 * @param   string    $slug       The slug to look for
	 * @param   int|null  $excludeId  Exclude the ticket with this ID. NULL to exclude none.
	 *
	 * @return  array
	 *
	 * @since   5.0.0
	 */
	private function getSimilarSlugs(string $slug, ?int $excludeId): array
	{
		$slug    = trim($slug);
		$altslug = $slug . '-%';
		/** @var DatabaseDriver $db */
		$db    = Factory::getContainer()->get(DatabaseInterface::class);
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select($db->quoteName('alias'))
			->from($db->quoteName('#__ats_tickets'))
			->where(
				'(' .
				'(' . $db->quoteName('alias') . ' LIKE :altslug' . ') OR ' .
				'(' . $db->quoteName('alias') . ' = :slug' . ')' .
				')'
			)
			->bind(':slug', $slug, ParameterType::STRING)
			->bind(':altslug', $altslug, ParameterType::STRING);

		if (is_numeric($excludeId) && ((int) $excludeId > 0))
		{
			$query
				->where($db->quoteName('id') . ' != :excludeId')
				->bind(':excludeId', $excludeId, ParameterType::INTEGER);
		}

		try
		{
			return $db->setQuery($query)->loadColumn() ?: [];
		}
		catch (Exception $e)
		{
			return [];
		}
	}

	private function filterByTableFields(array $array)
	{
		return array_filter($array, function ($x) {
			return !in_array($x, ['typeAlias', 'newTags', 'tagsHelper']) && $this->hasField($x);
		}, ARRAY_FILTER_USE_KEY);
	}
}