<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\View\Ticket;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Helper\Permissions;
use Akeeba\Component\ATS\Administrator\Helper\UserTags;
use Akeeba\Component\ATS\Administrator\Mixin\RunPluginsTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ViewLoadAnyTemplateTrait;
use Akeeba\Component\ATS\Administrator\Model\ManagernoteModel;
use Akeeba\Component\ATS\Administrator\Model\PostModel;
use Akeeba\Component\ATS\Administrator\Model\TicketModel;
use Akeeba\Component\ATS\Administrator\Table\TicketTable;
use Exception;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\View\GenericDataException;
use Joomla\CMS\MVC\View\HtmlView as BaseHtmlView;
use Joomla\CMS\Toolbar\ToolbarHelper;

class HtmlView extends BaseHtmlView
{
	use ViewLoadAnyTemplateTrait;
	use RunPluginsTrait;

	/**
	 * Privileges of the currently logged in user for this ticket
	 *
	 * @var   array{
	 *            view:bool,post:bool,delete:bool,edit:bool,"edit.state":bool,admin:bool,close:bool,attachment:bool,
	 *            "notes.read":bool,"notes.create":bool,"notes.edit.own":bool,"notes.edit":bool,"notes.delete":bool
	 *          }
	 * @since 5.0.0
	 */
	protected $canDo;

	/**
	 * The Form object
	 *
	 * @var    Form
	 * @since  5.0.0
	 */
	protected $form;

	/**
	 * The active item
	 *
	 * @var    TicketTable
	 * @since  5.0.0
	 */
	protected $item;

	/**
	 * The form to create a new manager's note
	 *
	 * @var   Form|null
	 * @since 5.0.0
	 */
	protected $managerNoteForm;

	/**
	 * The form to create a new ticket reply
	 *
	 * @var   Form|null
	 * @since 5.0.0
	 */
	protected $replyForm;

	/**
	 * The model state
	 *
	 * @var    object
	 * @since  5.0.0
	 */
	protected $state;

    protected $invited_users;

	/** @inheritdoc */
	public function display($tpl = null)
	{
		/** @var TicketModel $model */
		$model               = $this->getModel();
		$this->item          = $model->getItem();
		$this->form          = $model->getForm($this->item->getProperties());
		$this->state         = $model->getState();
        $this->invited_users = $model->getInvitedUsers();

		// Check for errors.
		if (method_exists($this->getModel(), 'getErrors'))
		{
			/** @noinspection PhpDeprecationInspection */
			$errors = $this->getModel()->getErrors();

			if (is_countable($errors) && count($errors))
			{
				throw new GenericDataException(implode("\n", $errors), 500);
			}
		}

		$this->canDo = Permissions::getTicketPrivileges($this->item);

		if (!empty($this->item->id) && $this->canDo['post'])
		{
			$userTags = (defined('ATS_PRO') && ATS_PRO)
				? UserTags::getUserTags($this->item->created_by)
				: [];

			/** @var PostModel $model */
			$postModel       = $this->getModel('post');
			$this->replyForm = $postModel->getForm([
				'ticket_id' => $this->item->getId(),
				'usertags'  => $userTags,
			]);
		}

		$canCreateNote = $this->canDo['admin'] || $this->canDo['notes.create'];

		if (!empty($this->item->id) && $canCreateNote && defined('ATS_PRO') && ATS_PRO)
		{
			/** @var ManagernoteModel $model */
			$managerNoteModel      = $this->getModel('managernote');
			$this->managerNoteForm = $managerNoteModel->getForm([
				'ticket_id' => $this->item->getId(),
			]);
		}

		// Set the layout for new tickets
		if (empty($this->item->id))
		{
			$this->setLayout('newticket');
		}

		// Set the category for Canned Replies permissions checks
		Factory::getApplication()->getSession()->set('com_ats.cannedreplies.category', $this->form->getValue('catid'));

		$this->addToolbar();

		parent::display($tpl);
	}

	/**
	 * Set up Joomla's toolbar.
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	protected function addToolbar(): void
	{
		Factory::getApplication()->getInput()->set('hidemainmenu', true);

		$isNew       = empty($this->item->id);
		$hasCategory = !empty($this->form->getValue('catid'));
		$cannotSave  = $isNew && !$hasCategory;

		ToolbarHelper::title(Text::_('COM_ATS_TITLE_TICKETS_' . (empty($this->item->id) ? 'ADD' : 'EDIT')), 'icon-ats');

		// New ticket can only have Save & Close and Cancel, not just Save
		if ($isNew && $this->canDo['post'] && !$cannotSave)
		{
			ToolbarHelper::save('ticket.save', 'COM_ATS_TICKETS_LBL_NEWTICKET_SUBMIT');
		}
		elseif (!$isNew)
		{
			ToolbarHelper::save('ticket.save');
			ToolbarHelper::apply('ticket.apply');
		}

		ToolbarHelper::cancel('ticket.cancel', $isNew ? 'JTOOLBAR_CANCEL' : 'JTOOLBAR_CLOSE');
	}

}
