<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Component\ATS\Administrator\View\Tickets;

use Akeeba\Component\ATS\Administrator\Mixin\ViewListLimitFixTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ViewLoadAnyTemplateTrait;
use Akeeba\Component\ATS\Administrator\Mixin\ViewTableUITrait;
use Akeeba\Component\ATS\Administrator\Mixin\ViewToolbarTrait;
use Akeeba\Component\ATS\Administrator\Model\TicketsModel;
use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Helper\ContentHelper;
use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\View\GenericDataException;
use Joomla\CMS\MVC\View\HtmlView as BaseHtmlView;
use Joomla\CMS\Pagination\Pagination;
use Joomla\CMS\Toolbar\Button\DropdownButton;
use Joomla\CMS\Toolbar\Toolbar;
use Joomla\CMS\Toolbar\ToolbarHelper;
use Joomla\Registry\Registry;

defined('_JEXEC') or die;

class HtmlView extends BaseHtmlView
{
	use ViewLoadAnyTemplateTrait;
	use ViewTableUITrait;
	use ViewListLimitFixTrait;
	use ViewToolbarTrait;

	/**
	 * The active search filters
	 *
	 * @var    array
	 * @since  5.0.0
	 */
	public $activeFilters = [];

	/**
	 * The search tools form
	 *
	 * @var    Form
	 * @since  5.0.0
	 */
	public $filterForm;

	/**
	 * Should I show the Time column?
	 *
	 * @var   bool
	 * @since 5.4.0
	 */
	public $showTimeSpent = true;

	/**
	 * An array of items
	 *
	 * @var    array
	 * @since  5.0.0
	 */
	protected $items = [];

	/**
	 * The pagination object
	 *
	 * @var    Pagination
	 * @since  5.0.0
	 */
	protected $pagination;

	/**
	 * The model state
	 *
	 * @var    Registry
	 * @since  5.0.0
	 */
	protected $state;

	/**
	 * Is this view an Empty State
	 *
	 * @var    boolean
	 * @since  5.0.0
	 */
	private $isEmptyState = false;

	/** @inheritdoc */
	public function display($tpl = null)
	{
		/** @var TicketsModel $model */
		$model               = $this->getModel();
		$this->fixListLimitPastTotal($model);
		$this->items         = $model->getItems();
		$this->pagination    = $model->getPagination();
		$this->state         = $model->getState();
		$this->filterForm    = $model->getFilterForm();
		$this->activeFilters = $model->getActiveFilters();
		$this->showTimeSpent = ComponentHelper::getParams('com_ats')->get('timespent_hide', 1) != 1;

		// Check for errors.
		if (method_exists($this->getModel(), 'getErrors'))
		{
			/** @noinspection PhpDeprecationInspection */
			$errors = $this->getModel()->getErrors();

			if (is_countable($errors) && count($errors))
			{
				throw new GenericDataException(implode("\n", $errors), 500);
			}
		}

		if (!\count($this->items) && $this->isEmptyState = $this->getModel()->getIsEmptyState())
		{
			$this->setLayout('emptystate');
		}

		$this->addToolbar();

		// We do not need to filter by language when multilingual is disabled
		if (!Multilanguage::isEnabled())
		{
			unset($this->activeFilters['language']);
			$this->filterForm->removeField('language', 'filter');
		}

		parent::display($tpl);
	}

	public function timeSpentHumanReadable(?float $timeSpent): string
	{
		if ($timeSpent < 0.001)
		{
			return '<span class="text-muted">00</span><span class="text-muted small">:00</span>';
		}

		$hours     = intval(floor($timeSpent / 60));
		$timeSpent -= $hours * 60;
		$minutes   = floor($timeSpent);
		$seconds   = intval(ceil(60 * ($timeSpent - $minutes)));

		if ($hours > 0)
		{
			return sprintf('<span class="text-danger-emphasis">%02d</span>:<span class="text-info-emphasis">%02d</span><span class="text-muted small">:%02d</span>', $hours, $minutes, $seconds);
		}

		return sprintf('<span class="text-info-emphasis">%02d</span><span class="text-muted small">:%02d</span>', $minutes, $seconds);
	}

	/**
	 * Set up Joomla's toolbar.
	 *
	 * @return  void
	 * @throws  Exception
	 * @since   5.0.0
	 */
	private function addToolbar()
	{
		$canDo = ContentHelper::getActions('com_ats', 'category', $this->state->get('filter.catid'));
		$user = Factory::getApplication()->getIdentity();

		// Get the toolbar object instance
		$toolbar = $this->getToolbarCompat();

		ToolbarHelper::title(sprintf(Text::_('COM_ATS_TITLE_TICKETS')), 'icon-ats');

		if ($canDo->get('core.create') || \count($user->getAuthorisedCategories('com_ats', 'core.create')) > 0)
		{
			$toolbar->addNew('ticket.add');
		}

		if (!$this->isEmptyState && $canDo->get('core.edit.state'))
		{
			/** @var DropdownButton $dropdown */
			$dropdown = $toolbar->dropdownButton('status-group')
				->text('JTOOLBAR_CHANGE_STATUS')
				->toggleSplit(false)
				->icon('icon-ellipsis-h')
				->buttonClass('btn btn-action')
				->listCheck(true);

			$childBar = $dropdown->getChildToolbar();

			$childBar->publish('tickets.publish')->listCheck(true);

			$childBar->unpublish('tickets.unpublish')->listCheck(true);

			$childBar->standardButton('makeprivate')
				->text('COM_ATS_TICKETS_PUBLIC_MAKE_PRIVATE')
				->task('tickets.makeprivate')
				->icon('fa fa-eye-slash')
				->listCheck(true);
			$childBar->standardButton('makepublic')
				->text('COM_ATS_TICKETS_PUBLIC_MAKE_PUBLIC')
				->task('tickets.makepublic')
				->icon('fa fa-eye')
				->listCheck(true);

			// Add a batch button
			if ($user->authorise('core.create', 'com_ats')
				&& $user->authorise('core.edit', 'com_ats')
				&& $user->authorise('core.edit.state', 'com_ats'))
			{
				$childBar->popupButton('batch')
					->text('JTOOLBAR_BATCH')
					->selector('collapseModal')
					->listCheck(true);
			}
		}

		if (!$this->isEmptyState && $canDo->get('core.delete'))
		{
			$toolbar->delete('tickets.delete')
				->text('JTOOLBAR_DELETE')
				->message('JGLOBAL_CONFIRM_DELETE')
				->listCheck(true);
		}

		if ($user->authorise('core.admin', 'com_ats') || $user->authorise('core.options', 'com_ats'))
			$toolbar->preferences('com_ats');
		{
		}

		ToolbarHelper::back('COM_ATS_TITLE_CPANEL', 'index.php?option=com_ats');
	}
}
