<?php
/**
 * @package   ats
 * @copyright Copyright (c)2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Utility\Utility;

/**
 * Workaround for Joomla core bug.
 *
 * Joomla provides a core field type for uploads (the file type, i.e. \Joomla\CMS\Form\Field\FileField). Its default
 * layout (`joomla.form.field.file` i.e. the `layouts/joomla/form/field/file.php` file) displays the maximum upload
 * filesize, but the value is displays IS WRONG!
 *
 * The core layout file ERRONEOUSLY calls `\Joomla\CMS\Utility\Utility::getMaxUploadSize` WITHOUT any parameters.
 * Therefore, it only gets the minimum between the PHP configuration values `post_max_size` and `upload_max_filesize`.
 *
 * However, Joomla allows core (e.g. com_media) and third party components to provide a component-level configuration
 * parameter `upload_maxsize` where the user can specify the maximum upload size (in megabytes) for files uploaded with
 * this specific extension. This is used by Joomla's `\Joomla\CMS\Helper\MediaHelper::canUpload` method. This method
 * is the **CANONICAL AND MANDATED** way for all extensions, core and third party, to determine if a user-uploaded file
 * should be allowed to be uploaded.
 *
 * This creates a serious discrepancy when the `upload_maxsize` component option is LOWER THAN both of the PHP
 * configuration values `post_max_size` and `upload_max_filesize`.
 *
 * Here's an example.
 *
 * PHP is configured with `post_max_size` 80 MiB and `upload_max_filesize` 32 MiB. ATS is configured with
 * `upload_maxsize` 10 MiB.
 *
 * The ACTUAL maximum upload filesize, as determined BY JOOMLA ITSELF in `\Joomla\CMS\Helper\MediaHelper::canUpload` is
 * the lowest of these values i.e. 10 MiB.
 *
 * The `layouts/joomla/form/field/file.php` ignores the component configuration. Therefore, it reports the lowest value
 * between `post_max_size` and `upload_maxsize` which is 32 MiB.
 *
 * A user seeing in the interface that the maximum upload size is 32 MiB tries to upload a 12 MiB file. Their upload is
 * rejected because it's bigger than the ACTUAL maximum upload filesize of 10 MiB. This leaves the user confused, and
 * the site owner blames ATS.
 *
 * However, no ATS code has been involved! It's Joomla reporting the wrong maximum upload size (without any way for us
 * to rectify it), and it's Joomla applying the different (but correct!) maximum upload size in its file upload code!
 *
 * This issue was reported by a client of ours to Joomla. Joomla refused to even acknowledge that the issue exists
 * even though it's blatantly obvious and there were detailed instructions to locate and fix it. So, we have to override
 * Joomla core code because Joomla won't fix its core bugs. You can't make that stuff up!
 */

extract($displayData);

/**
 * Layout variables
 * -----------------
 * @var   string   $autocomplete    Autocomplete attribute for the field.
 * @var   boolean  $autofocus       Is autofocus enabled?
 * @var   string   $class           Classes for the input.
 * @var   string   $description     Description of the field.
 * @var   boolean  $disabled        Is this field disabled?
 * @var   string   $group           Group the field belongs to. <fields> section in form XML.
 * @var   boolean  $hidden          Is this field hidden in the form?
 * @var   string   $hint            Placeholder for the field.
 * @var   string   $id              DOM id of the field.
 * @var   string   $label           Label of the field.
 * @var   string   $labelclass      Classes to apply to the label.
 * @var   boolean  $multiple        Does this field support multiple values?
 * @var   string   $name            Name of the input field.
 * @var   string   $onchange        Onchange attribute for the field.
 * @var   string   $onclick         Onclick attribute for the field.
 * @var   string   $pattern         Pattern (Reg Ex) of value of the form field.
 * @var   boolean  $readonly        Is this field read only?
 * @var   boolean  $repeat          Allows extensions to duplicate elements.
 * @var   boolean  $required        Is this field required?
 * @var   integer  $size            Size attribute of the input.
 * @var   boolean  $spellcheck      Spellcheck state for the form field.
 * @var   string   $validate        Validation rules to apply.
 * @var   string   $value           Value attribute of the field.
 * @var   array    $checkedOptions  Options that will be set as checked.
 * @var   boolean  $hasValue        Has this field a value assigned?
 * @var   array    $options         Options available for this field.
 * @var   array    $inputType       Options available for this field.
 * @var   string   $accept          File types that are accepted.
 * @var   string   $dataAttribute   Miscellaneous data attributes preprocessed for HTML output
 * @var   array    $dataAttributes  Miscellaneous data attribute for eg, data-*
 */

$custom  = ComponentHelper::getParams('com_ats')->get('upload_maxsize', 10);
$custom  = intval($custom ?? 0) <= 0 ? null : $custom;
$maxSize = HTMLHelper::_('number.bytes', Utility::getMaxUploadSize($custom));

?>
	<input type="file"
		   name="<?php echo $name; ?>"
		   id="<?php echo $id; ?>"
			<?php echo !empty($size) ? ' size="' . $size . '"' : ''; ?>
			<?php echo !empty($accept) ? ' accept="' . $accept . '"' : ''; ?>
			<?php echo !empty($class) ? ' class="form-control ' . $class . '"' : ' class="form-control"'; ?>
			<?php echo !empty($multiple) ? ' multiple' : ''; ?>
			<?php echo $disabled ? ' disabled' : ''; ?>
			<?php echo $autofocus ? ' autofocus' : ''; ?>
			<?php echo $dataAttribute; ?>
			<?php echo !empty($onchange) ? ' onchange="' . $onchange . '"' : ''; ?>
			<?php echo $required ? ' required' : ''; ?>><br>
<?php echo Text::sprintf('JGLOBAL_MAXIMUM_UPLOAD_SIZE_LIMIT', $maxSize); ?>