/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

"use strict";

// Handle the ticket status dropdown
[].slice.call(document.querySelectorAll(".atsStatusDropdown")).forEach(function (el)
{
    el.addEventListener("click", function (event)
    {
        event.preventDefault();

        var elTarget = event.currentTarget;

        if (!elTarget)
        {
            return;
        }

        var status    = elTarget.dataset.status;
        var statuses  = Joomla.getOptions("com_ats.statuses", {});
        var allColors = Joomla.getOptions("com_ats.allStatusColors", {})

        if (!statuses[status])
        {
            return;
        }

        var ticketid   = elTarget.dataset.ticketid;
        var dropdownId = elTarget.dataset.dropdownid;
        var elButton   = document.getElementById(dropdownId);

        if (!elButton)
        {
            return;
        }

        elButton.classList.add("disabled");

        var systemPaths = Joomla.getOptions("system.paths");
        var url         = (systemPaths ? systemPaths.root : window.location.pathname) +
            "?option=com_ats&task=ticket.ajax_set_status&format=json&id="
            + encodeURIComponent(ticketid) + "&status=" + encodeURIComponent(status) + "&" +
            encodeURIComponent(Joomla.getOptions("csrf.token")) + "=1";

        Joomla.request({
            url:       url,
            method:    "POST",
            headers:   {"Content-Type": "application/json"},
            perform:   true,
            onSuccess: function (rawJson)
                       {
                           elButton.classList.remove("disabled");

                           var response = JSON.parse(rawJson);

                           if (!response.success)
                           {
                               Joomla.renderMessages({error: [response.message]});

                               return;
                           }

                           elButton.innerText = statuses[status]["label"];
                           allColors.forEach(function (oldColor)
                           {
                               elButton.classList.remove(oldColor);
                           });
                           elButton.classList.add(statuses[status]["color"]);
                       },
            onError:   function (xhr)
                       {
                           elButton.classList.remove("disabled");

                           var rawJson = xhr.responseText;

                           if (rawJson)
                           {
                               var data = JSON.parse(rawJson);

                               if (data.error && data.message)
                               {
                                   Joomla.renderMessages({error: [data.message]});

                                   return;
                               }
                           }

                           Joomla.renderMessages({error: ["AJAX Error"]});
                       }
        })
    });
});

// Handle the assigned manager dropdown
[].slice.call(document.querySelectorAll(".atsAssignedDropdown")).forEach(function (el)
{
    el.addEventListener("click", function (event)
    {
        event.preventDefault();

        var elTarget = event.currentTarget;

        if (!elTarget)
        {
            return;
        }

        var assigned  = elTarget.dataset.assigned;
        var managers  = Joomla.getOptions("com_ats.managers", {});
        var allColors = Joomla.getOptions("com_ats.allManagerColors", {})

        if (!managers[assigned])
        {
            return;
        }

        var ticketid   = elTarget.dataset.ticketid;
        var dropdownId = elTarget.dataset.dropdownid;
        var elButton   = document.getElementById(dropdownId);

        if (!elButton)
        {
            return;
        }

        elButton.classList.add("disabled");

        var systemPaths = Joomla.getOptions("system.paths");
        var url         = (systemPaths ? systemPaths.root : window.location.pathname) +
            "?option=com_ats&task=ticket.ajax_set_assigned&format=json&id="
            + encodeURIComponent(ticketid) + "&assigned=" + encodeURIComponent(assigned) + "&" +
            encodeURIComponent(Joomla.getOptions("csrf.token")) + "=1";

        Joomla.request({
            url:       url,
            method:    "POST",
            headers:   {"Content-Type": "application/json"},
            perform:   true,
            onSuccess: function (rawJson)
                       {
                           elButton.classList.remove("disabled");

                           var response = JSON.parse(rawJson);

                           if (!response.success)
                           {
                               Joomla.renderMessages({error: [response.message]});

                               return;
                           }

                           elButton.innerText = managers[assigned]["name"];
                           allColors.forEach(function (oldColor)
                           {
                               elButton.classList.remove(oldColor);
                           });
                           elButton.classList.add(managers[assigned]["color"]);
                       },
            onError:   function (xhr)
                       {
                           elButton.classList.remove("disabled");

                           var rawJson = xhr.responseText;

                           if (rawJson)
                           {
                               var data = JSON.parse(rawJson);

                               if (data.error && data.message)
                               {
                                   Joomla.renderMessages({error: [data.message]});

                                   return;
                               }
                           }

                           Joomla.renderMessages({error: ["AJAX Error"]});
                       }
        })
    });
});

// Handle autocollapsible button labels
[].slice.call(document.querySelectorAll(".atsAutoCollapse")).forEach(function (el)
{
    var atsAutoCollapseHandler = function (event, add)
    {
        var elButton = event.currentTarget;
        var idLabel  = elButton.dataset.autocollapseTarget;
        var elLabel  = idLabel ? document.getElementById(idLabel) : null;

        if (!elLabel)
        {
            return;
        }

        if (add)
        {
            elLabel.classList.add("visually-hidden");
        }
        else
        {
            elLabel.classList.remove("visually-hidden");
        }
    };

    el.addEventListener("mouseenter", function (event)
    {
        atsAutoCollapseHandler(event, false);
    });

    el.addEventListener("focus", function (event)
    {
        atsAutoCollapseHandler(event, false);
    });

    el.addEventListener("mouseleave", function (event)
    {
        atsAutoCollapseHandler(event, true);
    });

    el.addEventListener("blur", function (event)
    {
        atsAutoCollapseHandler(event, true);
    });
});

// Handle ticket / reply area overlays because Joomla and Bootstrap collapsible seem to be mortal enemies...
[].slice.call(document.querySelectorAll(".atsReplyOverlayAcknowledge")).forEach(function (el)
{
    el.addEventListener("click", function (event)
    {
        var elButton    = event.currentTarget;
        var elTarget    = document.getElementById(elButton.dataset.target ?? "none");
        var elContainer = document.getElementById(elButton.dataset.container ?? "none");

        if (!elTarget)
        {
            return;
        }

        elTarget.classList.remove("d-none");

        if (!elContainer)
        {
            return;
        }

        elContainer.classList.add("d-none");
    });
});

// Handle the change of category in the new ticket page
[].slice.call(document.querySelectorAll(".atsNewTicketChangeCategory")).forEach(function (el)
{
    el.addEventListener("change", function (event)
    {
        var elForm     = document.forms.adminForm;
        var elTask     = document.forms.adminForm.task;

        if (!elForm || !elTask)
        {
            return;
        }

        elTask.value = 'reload';

        var elUrl = Joomla.getOptions('com_ats_newticket_reloadurl', '');

        if (elUrl !== '')
        {
            elForm.action = elUrl;
        }

        elForm.submit();
    });
});

[].slice.call(document.querySelectorAll(".atsInstantSearch")).forEach(function (el)
{
    el.addEventListener("keyup", function (event)
    {
        var elInput     = event.currentTarget;
        var elContainer = document.getElementById(elInput.id + "_searchresults");

        if (!elContainer)
        {
            return;
        }

        var oldSearchQuery = elInput.dataset.atsSearchQuery ?? "";
        var oldTimerId     = 1 * (elInput.dataset.atsSearchTimerId ?? "0");
        var options        = Joomla.getOptions("ats.instantsearch.options", {
            method: "none",
            url:    "https://duckduckgo.com/search.html?site={site}&q={search}",
            filter: 0,
            menu:   0
        });

        if (oldTimerId)
        {
            window.clearTimeout(oldTimerId);
        }

        // OK, let's make sure we have enough things to search for
        var searchQuery = elInput.value;

        if ((searchQuery === "") || (searchQuery.length < 2))
        {
            elContainer.classList.add("d-none");

            return;
        }

        if (oldSearchQuery === searchQuery)
        {
            return;
        }

        elInput.dataset.atsSearchQuery     = searchQuery;
        elInput.dataset.atsSearchTimerId = window.setTimeout(function() {
            elContainer.innerHTML = "";
            elContainer.classList.remove("d-none");

            var elIframe = document.createElement("iframe");
            elIframe.setAttribute("referrerpolicy", "unsafe-url");
            elIframe.src = options.url
                                  .replace("{search}", encodeURIComponent(searchQuery))
                                  .replace("{site}", encodeURIComponent(options.host ?? ""));

            elContainer.appendChild(elIframe);
        }, 250);
    });
});

[].slice
    .call(document.querySelectorAll('.ats-ticket-new > form#adminForm, .ats-ticket form#replyForm'))
    .forEach((el) => {
        el.addEventListener('submit', (event) => {
            const elSubmitDialog = document.getElementById('ats-onsubmit-overlay');

            if (!elSubmitDialog) {
                return;
            }

            elSubmitDialog.style.display = 'block';
            window.scrollTo({
                top: 0,
                behavior: "smooth"
            });
        });
    });