<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Plugin\Privacy\ATS\Extension;

defined('_JEXEC') or die;

use Akeeba\Component\ATS\Administrator\Table\AttachmentTable;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\CMS\User\User;
use Joomla\Component\Privacy\Administrator\Export\Domain;
use Joomla\Component\Privacy\Administrator\Plugin\PrivacyPlugin;
use Joomla\Component\Privacy\Administrator\Table\RequestTable as PrivacyTableRequest;
use Joomla\Event\Event;

class AkeebaTicketSystem extends PrivacyPlugin
{
	/**
	 * Disallow registering legacy listeners since we use SubscriberInterface
	 *
	 * @var   bool
	 * @since 5.2.6
	 */
	protected $allowLegacyListeners = false;

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return  array
	 *
	 * @since   5.2.6
	 */
	public static function getSubscribedEvents(): array
	{
		if (!ComponentHelper::isEnabled('com_ats'))
		{
			return [];
		}

		return [
			'onPrivacyExportRequest' => 'onPrivacyExportRequest',
			'onPrivacyRemoveData'    => 'onPrivacyRemoveData',
		];
	}

	/**
	 * Processes an export request for Joomla core user data
	 *
	 * This event will collect data for the following core tables:
	 *
	 * - #__ats_tickets
	 * - #__ats_posts
	 * - #__ats_attachments
	 * - #__ats_managernotes
	 *
	 * @param   Event  $event
	 *
	 * @return  void
	 * @since   5.2.6
	 *
	 * @noinspection PhpUnused
	 */
	public function onPrivacyExportRequest(Event $event): void
	{
		/**
		 * @var   PrivacyTableRequest $request The request record being processed
		 * @var   User                $user    The user account associated with this request if available
		 */
		[$request, $user] = array_values($event->getArguments());
		$result = $event->getArgument('result', []);

		if (empty($user->id ?? null) || $user->id <= 0)
		{
			return;
		}

		$ret = array_filter(
			[
				$this->getTickets($user->id),
				$this->getTicketFields($user->id),
				$this->getPosts($user->id),
				$this->getAttachments($user->id),
				$this->getManagerNotes($user->id),
			],
			fn($x) => !empty($x) && count($x->getItems()) > 0
		);

		$event->setArgument('result', array_merge($result, [$ret]));
	}

	/**
	 * Removes the data associated with a remove information request
	 *
	 * @param   Event  $event
	 *
	 * @return  void
	 * @since   5.2.6
	 *
	 * @noinspection PhpUnused
	 */
	public function onPrivacyRemoveData(Event $event): void
	{
		/**
		 * @var   PrivacyTableRequest $request The request record being processed
		 * @var   User                $user    The user account associated with this request if available
		 */
		[$request, $user] = array_values($event->getArguments());

		$language = $this->getApplication()->getLanguage();
		$language->load('com_ats', JPATH_ADMINISTRATOR);
		$language->load('com_ats', JPATH_SITE);

		/** @var MVCFactoryInterface $factory */
		$factory = $this->getApplication()->bootComponent('com_ats')->getMvcFactory();
		$db      = $this->getDatabase();

		$selectQuery = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('*')
			->from($db->quoteName('#__ats_attachments'))
			->where($db->quoteName('created_by') . ' = :userId')
			->bind(':userId', $userId);

		// Remove and pseudonymise attachments
		/** @var AttachmentTable $attachmentTable */
		$attachmentTable = $factory->createTable('Attachment', 'Administrator');

		foreach ($db->setQuery($selectQuery)->getIterator() as $record)
		{
			$attachmentTable->reset();
			$attachmentTable->id = null;

			$attachmentTable->bind($record);

			if ($attachmentTable->getAbsoluteFilename())
			{
				@unlink($attachmentTable->getAbsoluteFilename());
			}

			$attachmentTable->save([
				'original_filename' => 'pseudonymised_' . hash('md5', random_bytes(16)),
			]);
		}

		// Pseudonymize manager notes
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->update($db->quoteName('#__ats_managernotes'))
			->set(
				$db->quoteName('content_html') . ' = ' . $db->quote(
					'<p>' . Text::_('PLG_PRIVACY_ATS_REMOVED') . '</p>'
				)
			)
			->where($db->quoteName('created_by') . ' = :userId')
			->bind(':userId', $userId);
		$db->setQuery($query)->execute();

		// Pseudonymize posts
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->update($db->quoteName('#__ats_posts'))
			->set(
				$db->quoteName('content_html') . ' = ' . $db->quote(
					'<p>' . Text::_('PLG_PRIVACY_ATS_REMOVED') . '</p>'
				)
			)
			->where($db->quoteName('created_by') . ' = :userId')
			->bind(':userId', $userId);
		$db->setQuery($query)->execute();

		// Pseudonymize, disable, and close tickets
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->update($db->quoteName('#__ats_tickets'))
			->set(
				[
					$db->quoteName('title') . ' = ' . $db->quote(Text::_('PLG_PRIVACY_ATS_REMOVED')),
					$db->quoteName('alias') . ' = CONCAT_WS(' . $db->quote('-') . ',' . $db->quote('removed') . ',' . $db->quoteName('id') . ')',
					$db->quoteName('enabled') . ' = 0',
					$db->quoteName('status') . ' = ' . $db->quote('C'),
				]
			)
			->where($db->quoteName('created_by') . ' = :userId')
			->bind(':userId', $userId);
		$db->setQuery($query)->execute();
	}

	/**
	 * Get a user's tickets for export
	 *
	 * @param   int  $userId  The user
	 *
	 * @return  Domain  Their tickets
	 *
	 * @since   5.2.6
	 */
	private function getTickets(int $userId): Domain
	{
		$db = $this->getDatabase();

		$domain = $this->createDomain('ats_tickets', 'Support tickets, via Akeeba Ticket System');

		$selectQuery = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('*')
			->from($db->quoteName('#__ats_tickets'))
			->where($db->quoteName('created_by') . ' = :userId')
			->bind(':userId', $userId);

		foreach ($db->setQuery($selectQuery)->getIterator() as $record)
		{
			$domain->addItem(
				$this->createItemFromArray(
					(array) $record,
					$record->id
				)
			);
		}

		return $domain;
	}

	/**
	 * Get a user's ticket custom fields for export
	 *
	 * @param   int  $userId  The user
	 *
	 * @return  Domain  Their tickets
	 *
	 * @since   5.2.6
	 */
	private function getTicketFields(int $userId): Domain
	{
		$db = $this->getDatabase();

		$selectQuery = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('*')
			->from($db->quoteName('#__ats_tickets'))
			->where($db->quoteName('created_by') . ' = :userId')
			->bind(':userId', $userId);

		$items = $db->setQuery($selectQuery)->loadObjectList();

		return $this->createCustomFieldsDomain('com_ats.ticket', $items);
	}

	/**
	 * Get a user's posts for export
	 *
	 * @param   int  $userId  The user
	 *
	 * @return  Domain  Their posts
	 *
	 * @since   5.2.6
	 */
	private function getPosts(int $userId): Domain
	{
		$db = $this->getDatabase();

		$domain = $this->createDomain('ats_posts', 'Support ticket posts, via Akeeba Ticket System');

		$selectQuery = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('*')
			->from($db->quoteName('#__ats_posts'))
			->where($db->quoteName('created_by') . ' = :userId')
			->bind(':userId', $userId);

		foreach ($db->setQuery($selectQuery)->getIterator() as $record)
		{
			$domain->addItem(
				$this->createItemFromArray(
					(array) $record,
					$record->id
				)
			);
		}

		return $domain;
	}

	/**
	 * Get a user's attachments for export
	 *
	 * @param   int  $userId  The user
	 *
	 * @return  Domain  Their attachments
	 *
	 * @since   5.2.6
	 */
	private function getAttachments(int $userId): Domain
	{
		$db = $this->getDatabase();

		$domain = $this->createDomain('ats_attachments', 'Support ticket attachments, via Akeeba Ticket System');

		$selectQuery = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('*')
			->from($db->quoteName('#__ats_attachments'))
			->where($db->quoteName('created_by') . ' = :userId')
			->bind(':userId', $userId);

		foreach ($db->setQuery($selectQuery)->getIterator() as $record)
		{
			$domain->addItem(
				$this->createItemFromArray(
					(array) $record,
					$record->id
				)
			);
		}

		return $domain;
	}

	/**
	 * Get a user's manager notes for export
	 *
	 * @param   int  $userId  The user
	 *
	 * @return  Domain  Their manager notes
	 *
	 * @since   5.2.6
	 */
	private function getManagerNotes(int $userId): Domain
	{
		$db = $this->getDatabase();

		$domain = $this->createDomain('ats_managernotes', 'Manager notes, via Akeeba Ticket System');

		$selectQuery = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
			->select('*')
			->from($db->quoteName('#__ats_managernotes'))
			->where($db->quoteName('created_by') . ' = :userId')
			->bind(':userId', $userId);

		foreach ($db->setQuery($selectQuery)->getIterator() as $record)
		{
			$domain->addItem(
				$this->createItemFromArray(
					(array) $record,
					$record->id
				)
			);
		}

		return $domain;
	}
}