<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Plugin\Fields\SQLUser\Extension;

defined('_JEXEC') or die;

use DOMElement;
use Exception;
use Joomla\CMS\Access\Access;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\User\User;
use Joomla\Component\Fields\Administrator\Plugin\FieldsListPlugin;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\ParameterType;
use stdClass;

/**
 * Fields Sql with User Data Plugin
 *
 * @since  5.0.6
 */
class SQLUser extends FieldsListPlugin  implements DatabaseAwareInterface
{
	use DatabaseAwareTrait;

	/**
	 * Transforms the field into a DOM XML element and appends it as a child on the given parent.
	 *
	 * @param   stdClass    $field   The field.
	 * @param   DOMElement  $parent  The field node parent.
	 * @param   Form        $form    The form.
	 *
	 * @return  DOMElement
	 *
	 * @since   5.0.6
	 */
	public function onCustomFieldsPrepareDom($field, DOMElement $parent, Form $form)
	{
		$fieldNode = parent::onCustomFieldsPrepareDom($field, $parent, $form);

		if (!$fieldNode)
		{
			return $fieldNode;
		}

		$fieldNode->setAttribute('value_field', 'text');
		$fieldNode->setAttribute('key_field', 'value');

		return $fieldNode;
	}

	/**
	 * The save event.
	 *
	 * @param   string                   $context  The context
	 * @param   \Joomla\CMS\Table\Table  $item     The table
	 * @param   boolean                  $isNew    Is new item
	 * @param   array                    $data     The validated data
	 *
	 * @return  boolean
	 *
	 * @since   5.0.6
	 */
	public function onContentBeforeSave($context, $item, $isNew, $data = [])
	{
		// Only work on new SQL fields
		if ($context != 'com_fields.field' || !isset($item->type) || $item->type != 'sqluser')
		{
			return true;
		}

		// If we are not a super admin, don't let the user create or update a SQL field
		if (!Access::getAssetRules(1)->allow('core.admin', $this->app->getIdentity()->getAuthorisedGroups()))
		{
			if (method_exists($item, 'setError'))
			{
				/** @noinspection PhpDeprecationInspection */
				$item->setError(Text::_('PLG_FIELDS_SQLUSER_CREATE_NOT_POSSIBLE'));

				return false;
			}

			throw new \RuntimeException(Text::_('PLG_FIELDS_SQLUSER_CREATE_NOT_POSSIBLE'));
		}

		return true;
	}

	/**
	 * Returns an array of key values to put in a list from the given field.
	 *
	 * @param   stdClass  $field  The field.
	 *
	 * @return  array
	 *
	 * @since   5.0.6
	 */
	public function getOptionsFromField($field)
	{
		$data = parent::getOptionsFromField($field);

		$db             = $this->getDatabase();
		$processedQuery = $this->processQuery($field->fieldparams->get('query', '') ?: '');

		try
		{
			$items = $db->setQuery($processedQuery)->loadObjectList() ?: [];
		}
		catch (Exception $e)
		{
			return [];
		}

		foreach ($items as $item)
		{
			if (!isset($item->value) || !isset($item->text))
			{
				continue;
			}

			$data[$item->value] = $item->text;
		}

		return $data;
	}


	/**
	 * Prepares the field value.
	 *
	 * @param   string    $context  The context.
	 * @param   stdclass  $item     The item.
	 * @param   stdclass  $field    The field.
	 *
	 * @return  string
	 *
	 * @since   5.0.6
	 */
	public function onCustomFieldsPrepareField($context, $item, $field)
	{
		// Check if the field should be processed by us
		if (!$this->isTypeSupported($field->type))
		{
			return '';
		}

		// Merge the params from the plugin and field which has precedence
		$fieldParams = clone $this->params;
		$fieldParams->merge($field->fieldparams);

		// Proprocess the query
		$processedQuery = $this->processQuery($fieldParams->get('query', '') ?: '');
		$fieldParams->set('query', $processedQuery);

		// Get the path for the layout file
		$path = PluginHelper::getLayoutPath('fields', $this->_name, $field->type);

		// Render the layout
		ob_start();
		include $path;
		$output = ob_get_clean();

		// Return the output
		return $output;
	}

	/**
	 * Replace the variables in the query
	 *
	 * @param   string  $query  The SQL query to process
	 *
	 * @return  string
	 *
	 * @since   5.0.6
	 */
	private function processQuery(string $query): string
	{
		if (empty($query))
		{
			return $query;
		}

		$user = $this->app->getIdentity() ?: new User();

		$userValues = array_combine(
			array_map(
				function (string $key): string {
					return '{USER:' . strtoupper($key) . '}';
				},
				array_keys($this->userToArray($user))
			),
			array_map(
				function ($value): string {
					return is_scalar($value) ? $this->getDatabase()->escape($value) : '';
				},
				array_values($this->userToArray($user))
			)
		);

		$userProfile   = $user->guest ? [] : $this->getUserProfileKeys($user->id ?: 0);
		$profileValues = array_combine(
			array_map(
				function (string $key): string {
					return '{PROFILE:' . strtoupper($key) . '}';
				},
				array_keys($userProfile)
			),
			array_map(
				function ($value): string {
					return is_scalar($value) ? $this->getDatabase()->escape($value) : '';
				},
				array_values($userProfile)
			)
		);

		$replacements = array_merge($userValues, $profileValues);

		return str_ireplace(array_keys($replacements), array_values($replacements), $query);
	}

	/**
	 * Convertsa a Joomla User object into an array.
	 *
	 * @param   User  $user
	 *
	 * @return  array
	 * @since   5.4.0
	 */
	private function userToArray(User $user): array
	{
		$asArray = (array) $user;

		return array_filter($asArray, fn($x) => !empty($x) && !is_numeric($x) && ord(substr($x, 0, 1)) !== 0, ARRAY_FILTER_USE_KEY);
	}

	/**
	 * Get the user profile keys and values
	 *
	 * @param   int  $id  The user ID
	 *
	 * @return  array  The user profile keys and their values
	 *
	 * @since   5.0.6
	 */
	private function getUserProfileKeys(int $id): array
	{
		$db    = $this->getDatabase();
		$query = (method_exists($db, 'createQuery') ? $db->createQuery() : $db->getQuery(true))
		            ->select([
			            $db->quoteName('profile_key'),
			            $db->quoteName('profile_value'),
		            ])
		            ->from($db->quoteName('#__user_profiles'))
		            ->where($db->quoteName('user_id') . ' = :user_id')
		            ->bind(':user_id', $id, ParameterType::INTEGER);

		try
		{
			return $db->setQuery($query)->loadAssocList('profile_key', 'profile_value') ?: [];
		}
		catch (Exception $e)
		{
			return [];
		}
	}
}
