<?php
/**
 * @package   ats
 * @copyright Copyright (c)2011-2025 Nicholas K. Dionysopoulos / Akeeba Ltd
 * @license   GNU General Public License version 3, or later
 */

namespace Akeeba\Plugin\Content\ATSShowOn\Extension;

defined('_JEXEC') or die;

use Joomla\CMS\Form\Form;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\User\User;
use Joomla\Component\Fields\Administrator\Helper\FieldsHelper;
use Joomla\Component\Fields\Administrator\Model\FieldsModel;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;
use Joomla\Utilities\ArrayHelper;
use ReflectionException;
use SimpleXMLElement;

/**
 * A simple plugin to add a “show on” attribute to Akeeba Ticket System custom fields.
 *
 * Based on the JT Showon plugin by JoomTools
 *
 * @since 5.0.0
 *
 * Copyright notice and information of the original plugin (the following 3 lines)
 * Author:    Guido De Gobbis <support@joomtools.de>
 * Copyright: Copyright (c) 2020 JoomTools.de - All rights reserved.
 * License:   GNU General Public License version 3 or later
 *
 * ~~~~~ THIS IS DERIVATIVE WORK, NOT THE ORIGINAL WORK. ~~~~~
 * We (Akeeba Ltd) have heavily modified this plugin to fit the purpose of our ticket system software's custom fields.
 *
 * Please do not not contact the original author. For any issues with this derivative work please contact the maintainer
 * of this derivative, Akeeba Ltd, through our support ticket page if you are a subscriber or through our site's Contact
 * Us page if you are not a subscriber or have found a bug.
 */
class ATSShowOn extends CMSPlugin implements SubscriberInterface
{
	/**
	 * Array of ticket fields
	 *
	 * @var     array
	 * @since   5.0.0
	 */
	protected static $itemFields = [];

	/** @inheritdoc */
	protected $autoloadLanguage = true;

	public static function getSubscribedEvents(): array
	{
		return [
			'onContentPrepareForm'             => 'onContentPrepareForm',
			'onCustomFieldsBeforePrepareField' => 'onCustomFieldsBeforePrepareField',
		];
	}


	/**
	 * Adds the shown property to Akeeba Ticket System custom fields. Applies showon in ticket edit form.
	 *
	 * @param   Event  $event
	 *
	 * @return  void
	 * @throws  ReflectionException
	 * @since   5.0.0
	 */
	public function onContentPrepareForm(Event $event): void
	{
		/**
		 * @var Form         $form
		 * @var object|array $data
		 */
		[$form, $data] = array_values($event->getArguments());

		$context = $form->getName();

		/**
		 * Add the showon attribute when editing the field definitions in the backend of the site.
		 */
		if (substr($context, 0, 25) === 'com_fields.field.com_ats.')
		{
			$fieldParams = JPATH_PLUGINS . '/' . $this->_type . '/' . $this->_name . '/xml/showon.xml';
			$showonXml   = new SimpleXMLElement($fieldParams, 0, true);

			$form->setField($showonXml);

			return;
		}

		// Handle showon for custom fields in the ticket edit page
		if ($context !== 'com_ats.ticket')
		{
			return;
		}

		/**
		 * We process the showon attributes in a loop to address interdependencies.
		 *
		 * For example, field A shows when field B and field C have a value, field C displays when field D has a value.
		 * When we reach field C we decide it must always be hidden because field D has a value different than C's show
		 * on condition AND is always hidden. Therefore we remove field C from display.
		 *
		 * In the next iteration we reach field B and realise that field C is no longer included in the form. Therefore
		 * field B must no longer have showon.
		 */
		$fieldSets     = $form->getFieldsets('com_fields');
		$lastSignature = '';

		while (true)
		{
			$displayedFields = [];

			foreach ($fieldSets as $fieldSetInfo)
			{
				$fieldSet = $form->getFieldset($fieldSetInfo->name);

				foreach ($fieldSet as $field)
				{
					$displayedFields[] = $field->fieldname;
					$this->evaluateShowOn($field, $form, (object) $data);
				}
			}

			$thisSignature = hash('md5', implode(':', $displayedFields));

			if ($thisSignature === $lastSignature)
			{
				break;
			}

			$lastSignature = $thisSignature;
		}
	}

	/**
	 * Validates the showon value and disables the output of the field if necessary.
	 *
	 * This only applies for DISPLAYING values. For editing purposes the second part of onContentPrepareForm kicks in.
	 *
	 * @param   Event  $event
	 *
	 * @return  void
	 * @throws  ReflectionException
	 * @since   5.0.0
	 */
	public function onCustomFieldsBeforePrepareField(Event $event): void
	{
		/**
		 * @var  string $context
		 * @var  object $item
		 * @var  object $field
		 */
		[$context, $item, $field] = array_values($event->getArguments());

		if ($context !== 'com_ats.ticket')
		{
			return;
		}

		$showOnData = trim($field->fieldparams->get('showon', null) ?? '');

		if (empty($showOnData))
		{
			return;
		}

		$itemFields = $this->getCachedFields($context, $item);

		$showon       = [];
		$showon['or'] = explode('[OR]', $showOnData);

		if (!empty($showon['or']))
		{
			foreach ($showon['or'] as $key => $value)
			{
				if (stripos($value, '[AND]') !== false)
				{
					[$or, $and] = explode('[AND]', $value, 2);

					$showon['and']      = explode('[AND]', $and);
					$showon['or'][$key] = $or;
				}
			}
		}

		if (!empty($showon['and']))
		{
			foreach ($showon['and'] as $value)
			{
				[$fieldName, $fieldValue] = explode(':', $value);

				if (empty($itemFields[$fieldName]) || $itemFields[$fieldName]->rawvalue != $fieldValue)
				{
					$field->params->set('display', 0);

					return;
				}
			}
		}

		foreach ($showon['or'] as $value)
		{
			[$fieldName, $fieldValue] = explode(':', $value);

			$showFieldOr[] = (!empty($itemFields[$fieldName]) && $itemFields[$fieldName]->rawvalue == $fieldValue);
		}

		if (!in_array(true, $showFieldOr))
		{
			$field->params->set('display', '0');
		}
	}

	/**
	 * Evaluates the showon conditions for a field.
	 *
	 * If all of the fields referenced in showon are present in the form we take no action. Joomla's client–side code
	 * will take care showing/hiding fields for us.
	 *
	 * If any of the fields referenced in showon is missing we take an action depending on whether the showon condition
	 * evaluates to show or hide:
	 * - Show: Must always show; we blank out the showon attribute.
	 * - Hide: Must always hide; we remove the field.
	 *
	 * @param   FormField  $field  The custom field we are evaluating
	 * @param   Form       $form   The form it belongs to
	 * @param   object     $item   The ticket being edited
	 *
	 * @return  void
	 * @throws  ReflectionException
	 * @since   5.0.0
	 */
	private function evaluateShowOn(FormField $field, Form $form, object $item): void
	{
		$showOnData = trim($field->showon);

		if (empty($showOnData))
		{
			return;
		}

		$itemFields = $this->getCachedFields($form->getName(), $item);

		$showon       = [];
		$showon['or'] = explode('[OR]', $showOnData);

		if (!empty($showon['or']))
		{
			foreach ($showon['or'] as $key => $value)
			{
				if (stripos($value, '[AND]') !== false)
				{
					[$or, $and] = explode('[AND]', $value, 2);

					$showon['and']      = explode('[AND]', $and);
					$showon['or'][$key] = $or;
				}
			}
		}

		$shouldHide   = false;
		$hasAllFields = true;

		if (!empty($showon['and']))
		{
			foreach ($showon['and'] as $value)
			{
				[$fieldName, $fieldValue] = explode(':', $value);

				$hasAllFields = $hasAllFields && $form->getField($fieldName, 'com_fields') !== false;

				if (empty($itemFields[$fieldName]) || $itemFields[$fieldName]->rawvalue != $fieldValue)
				{
					$shouldHide = true;
				}
			}
		}

		foreach ($showon['or'] as $value)
		{
			[$fieldName, $fieldValue] = explode(':', $value);

			$hasAllFields = $hasAllFields && $form->getField($fieldName, 'com_fields') !== false;

			$showFieldOr[] = (!empty($itemFields[$fieldName])
			                  && ($itemFields[$fieldName]->rawvalue ?? null) == $fieldValue);
		}

		$shouldHide = $shouldHide || !in_array(true, $showFieldOr);

		/**
		 * The field has a non–empty shown attribute AND all the fields which control its display are present in the
		 * form.
		 *
		 * We need to let Joomla's showon frontend code decide when to show the field. We are not taking any action.
		 */
		if ($hasAllFields)
		{
			return;
		}

		/**
		 * The field needs to be ALWAYS displayed: one of the fields controlling its display is not present in the form.
		 *
		 * We blank out its showon attribute.
		 */
		if (!$shouldHide)
		{
			$field->showon = '';

			return;
		}

		/**
		 * The field needs to be ALWAYS hidden: one of the fields controlling its display is not present in the form.
		 *
		 * We remove the field from the form.
		 */
		$form->removeField($field->fieldname, 'com_fields');
	}

	/**
	 * Gets all custom fields for this item.
	 *
	 * This includes fields which are not visible to the user. It is only ever used internally for determining which
	 * fields to display.
	 *
	 * @param   string  $context  The form context e.g. com_ats.ticket
	 * @param   object  $item     The item (ticket) being edited
	 *
	 * @return  array
	 *
	 * @throws  ReflectionException
	 * @since   5.0.0
	 */
	private function getCachedFields(string $context, object $item): array
	{
		$uniqueItemId = hash('md5', $item->id ?? '');

		if (array_key_exists($uniqueItemId, self::$itemFields))
		{
			return self::$itemFields[$uniqueItemId];
		}

		// Pretend it's a backend Super User to force loading all fields, regardless of their access level
		$cmsApp = $this->getApplication();
		try
		{
			$refClass = new \ReflectionObject($cmsApp);
			$refProp  = $refClass->getProperty('name');

			if (version_compare(PHP_VERSION, '8.1.0', 'lt'))
			{
				$refProp->setAccessible(true);
			}

			$appName = $refProp->getValue($cmsApp);
			$refProp->setValue($cmsApp, 'administrator');
		}
		catch (ReflectionException $e)
		{
			// Something failed with the application object reflection.
			return [];
		}

		try
		{
			$user     = $this->getApplication()->getIdentity() ?? new User();
			$refUser  = new \ReflectionObject($user);
			$refProp2 = $refUser->getProperty('isRoot');

			if (version_compare(PHP_VERSION, '8.1.0', 'lt'))
			{
				$refProp2->setAccessible(true);
			}

			$isRoot = $refProp2->getValue($user);
			$refProp2->setValue($user, true);
		}
		catch (\Exception $e)
		{
			// Something failed with the user manipulation.
			$refProp->setValue($cmsApp, $appName);

			return [];
		}

		// Get the private copy of the Fields model from the FieldsHelper
		$model = $this->getFieldsCacheModel();

		if ($model !== null)
		{
			// We need to clear the model's internal cache and query to force it to load all fields
			$refModelObject = new \ReflectionObject($model);
			$refCache       = $refModelObject->getProperty('cache');

			if (version_compare(PHP_VERSION, '8.1.0', 'lt'))
			{
				$refCache->setAccessible(true);
			}

			$refCache->setValue($model, []);
			$refQuery = $refModelObject->getProperty('query');

			if (version_compare(PHP_VERSION, '8.1.0', 'lt'))
			{
				$refQuery->setAccessible(true);
			}

			$refQuery->setValue($model, null);

			self::$itemFields[$uniqueItemId] = ArrayHelper::pivot(FieldsHelper::getFields($context, $item), 'name');

			$refCache->setValue($model, []);
			$refQuery->setValue($model, null);
		}

		// This MUST always run to set the application back to its original state
		$refProp->setValue($cmsApp, $appName);
		$refProp2->setValue($user, $isRoot);

		if ($model === null)
		{
			return [];
		}

		return self::$itemFields[$uniqueItemId];
	}

	private function getFieldsCacheModel(): ?FieldsModel
	{
		try
		{
			$refFieldsHelper = new \ReflectionClass(FieldsHelper::class);
		}
		catch (\Exception $e)
		{
			return null;
		}

		if (version_compare(PHP_VERSION, '8.3.0', 'ge'))
		{
			try
			{
				$model = $refFieldsHelper->getStaticPropertyValue('fieldsCache');
			}
			catch (\Exception $e)
			{
				return null;
			}
		}
		else
		{
			try
			{
				$refFieldsCache = $refFieldsHelper->getProperty('fieldsCache');

				if (version_compare(PHP_VERSION, '8.1.0', 'lt'))
				{
					$refFieldsCache->setAccessible(true);
				}

				$model = $refFieldsCache->getValue();
			}
			catch (\Exception $e)
			{
				return null;
			}
		}

		return $model;
	}
}
